import time
import unittest
import warnings

from iotas.note import Note

warnings.filterwarnings("ignore", "version")


class Test(unittest.TestCase):

    def test_duplicate(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.excerpt = note.title
        note.favourite = False
        note.category = "category"

        dupe = note.duplicate()
        self.assertIsNotNone(dupe)
        self.assertEqual(dupe.content, "content")
        self.assertEqual(dupe.title, "Test Note Title")
        self.assertEqual(dupe.excerpt, "Test Note Title")
        self.assertEqual(dupe.favourite, False)
        self.assertEqual(dupe.category, "category")

    def test_flag_changed(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.saving = False
        self.assertFalse(note.dirty)
        note.flag_changed(update_last_modified=False)
        self.assertTrue(note.dirty)
        self.assertEqual(note.last_modified, 0)
        self.assertFalse(note.dirty_while_saving)

        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.saving = False
        self.assertFalse(note.dirty)
        note.flag_changed(update_last_modified=True)
        self.assertTrue(note.dirty)
        self.assertTrue(note.last_modified > 0)
        self.assertFalse(note.dirty_while_saving)

        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.saving = True
        note.flag_changed(update_last_modified=False)
        self.assertTrue(note.dirty)
        self.assertEqual(note.last_modified, 0)
        self.assertTrue(note.dirty_while_saving)

    def test_regenerate_excerpt(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        self.assertEqual(note.excerpt, "")
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "content")

        self.assertFalse(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "content")

        note.content = ""
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "")

        note.content = "# updated content"
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "updated content")

        note.content = "## Test Note Title\npost-title"
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "post-title")

        note.content = "- one\n+ two\n* three"
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "• one   • two   • three")

        note.content = "- [ ] one\n- [x] two"
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(note.excerpt, "☐ one   ☑ two")

        note.content = """long long long long long long long long long long long long long long
        long long long long long long long long long long long long long long long long long long
        long long long long long long long long long long long long long long long long long long
        long long long long long long long long long long long long long long long long long long
        long long long long long long long long long long long long long long long long long long
        long long long long long long long long long long long long long long long long long long
        """
        self.assertTrue(note.regenerate_excerpt())
        self.assertEqual(len(note.excerpt), note.EXCERPT_LENGTH)

    def test_repopulate_meta_from_sync_update(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.title_is_top_line = False

        modified = int(time.time())
        meta = {
            "title": "New Title",
            "category": "category",
            "modified": modified,
            "etag": "etag",
            "id": "remoteid",
        }
        note.repopulate_meta_from_sync_update(meta)
        self.assertEqual(note.title, "New Title")
        self.assertEqual(note.category, "category")
        self.assertEqual(note.last_modified, modified)
        self.assertEqual(note.etag, "etag")
        self.assertEqual(note.remote_id, "remoteid")

        note.title = "Reset Title"
        note.title_is_top_line = True
        note.repopulate_meta_from_sync_update(meta)
        self.assertEqual(note.title, "Reset Title")
        self.assertEqual(note.server_sanitised_title, "New Title")

    def test_update_from_remote_sync(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        modified = int(time.time())
        remote_update = {
            "title": "New Title",
            "content": "new content",
            "category": "category",
            "modified": modified,
            "etag": "etag",
            "readonly": True,
            "favorite": True,
        }

        note.update_from_remote_sync(remote_update)
        self.assertEqual(note.title, "New Title")
        self.assertEqual(note.content, "new content")
        self.assertEqual(note.excerpt, "new content")
        self.assertEqual(note.category, "category")
        self.assertEqual(note.last_modified, modified)
        self.assertEqual(note.etag, "etag")
        self.assertTrue(note.favourite)
        self.assertTrue(note.read_only)

        remote_update["readonly"] = False
        remote_update["favorite"] = False
        note.update_from_remote_sync(remote_update)
        self.assertFalse(note.favourite)
        self.assertFalse(note.read_only)

    def test_update_title_from_top_line(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.title_is_top_line = True
        note.update_title_from_top_line()
        self.assertEqual(note.title, "content")

        note.content = "content"
        note.title = "Test Note Title"
        note.title_is_top_line = False
        note.update_title_from_top_line()
        self.assertEqual(note.title, "Test Note Title")

    def test_identical_excepting_sync_meta(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.category = "category"
        note.favourite = True
        note.locally_deleted = False
        note.remote_id = 7
        note.last_modified = int(time.time())
        note.etag = "etag"
        note.dirty = True

        second = Note(new_note=True)
        second.content = "content"
        second.title = "Test Note Title"
        second.category = "category"
        second.favourite = True
        second.locally_deleted = False
        second.remote_id = 7
        second.last_modified = 0
        second.etag = "otheretag"
        second.dirty = False

        self.assertTrue(note.identical_excepting_sync_meta(second))
        second.content = "changed"
        self.assertFalse(note.identical_excepting_sync_meta(second))

    def test_identical_to(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.category = "category"
        note.favourite = True
        note.locally_deleted = False
        note.remote_id = 7
        note.last_modified = int(time.time())
        note.etag = "etag"
        note.dirty = True

        second = Note(new_note=True)
        second.content = "content"
        second.title = "Test Note Title"
        second.category = "category"
        second.favourite = True
        second.locally_deleted = False
        second.remote_id = 7
        second.last_modified = note.last_modified
        second.etag = "etag"
        second.dirty = True

        self.assertTrue(note.identical_to(second))
        second.dirty = False
        self.assertFalse(note.identical_to(second))

    def test_etag_is_recent(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        self.assertFalse(note.etag_is_recent(""))
        note.etag = "etag"

        self.assertTrue(note.etag_is_recent("etag"))
        self.assertFalse(note.etag_is_recent("missing"))
        self.assertFalse(note.etag_is_recent(""))

        note.etag = "second"
        self.assertTrue(note.etag_is_recent("second"))
        self.assertTrue(note.etag_is_recent("etag"))

        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"
        note.etag = ""

        self.assertFalse(note.etag_is_recent(""))

    def test_from_backup(self) -> None:
        modified = int(time.time())
        details = {
            "Title": "New Title",
            "Category": "category",
            "Favourite": True,
            "LastModified": modified,
            "LocallyDeleted": False,
            "Dirty": False,
            "ETag": "etag",
            "RemoteId": 7,
        }
        note = Note.from_backup(details, "content", include_sync_attributes=False)
        self.assertIsNotNone(note)
        assert note  # mypy
        self.assertEqual(note.title, "New Title")
        self.assertEqual(note.content, "content")
        self.assertEqual(note.excerpt, "content")
        self.assertEqual(note.category, "category")
        self.assertEqual(note.last_modified, modified)
        self.assertEqual(note.favourite, True)
        self.assertEqual(note.read_only, False)

        note = Note.from_backup(details, "content", include_sync_attributes=True)
        self.assertIsNotNone(note)
        assert note  # mypy
        self.assertEqual(note.etag, "etag")
        self.assertEqual(note.remote_id, 7)

    def test_from_nextcloud(self) -> None:
        modified = int(time.time())
        remote = {
            "title": "New Title",
            "content": "new content",
            "category": "category",
            "modified": modified,
            "etag": "etag",
            "readonly": True,
            "favorite": True,
            "id": 7,
        }
        note = Note.from_nextcloud(remote)

        self.assertIsNotNone(note)
        self.assertEqual(note.title, "New Title")
        self.assertEqual(note.content, "new content")
        self.assertEqual(note.excerpt, "new content")
        self.assertEqual(note.category, "category")
        self.assertEqual(note.last_modified, modified)
        self.assertEqual(note.etag, "etag")
        self.assertEqual(note.read_only, True)
        self.assertEqual(note.favourite, True)
        self.assertEqual(note.remote_id, 7)

    def test_new_and_empty(self) -> None:
        note = Note(new_note=True)
        note.content = ""
        self.assertTrue(note.new_and_empty)

        note.content = "content"
        self.assertFalse(note.new_and_empty)

        note.content = ""
        note.id = 7
        self.assertFalse(note.new_and_empty)

    def test_has_id(self) -> None:
        note = Note(new_note=True)
        self.assertFalse(note.has_id)
        note.id = 13
        self.assertTrue(note.has_id)

    def test_has_remote_id(self) -> None:
        note = Note(new_note=True)
        self.assertFalse(note.has_remote_id)
        note.remote_id = 13
        self.assertTrue(note.has_remote_id)

    def test_content_loaded(self) -> None:
        note = Note(new_note=True)
        self.assertFalse(note.content_loaded)
        note.content = ""
        self.assertTrue(note.content_loaded)
