package HiD::Generator::BibtexPage;
# ABSTRACT: HiD Bibtex publication list page generator
use Moose;
with 'HiD::Generator';
use HiD::Page;
use BibTeX::Parser;
use IO::File;

our $VERSION = "0.4";


# a map of bibtex publication item names to real/speaking names
my $type_map = {
  'inproceedings' => 'Conference',
  'phdthesis'     => 'PhD Thesis',
  'article'       => 'Journal',
  'misc'          => 'Misc',
  'mastersthesis' => 'Diploma Thesis',
  'unpublished'   => 'Unpublished',
  'inbook'        => 'Chapter in Book',
  'incollection'  => 'Paper Collection'
};


sub generate {
  my( $self , $site ) = @_;
  return unless $site->config->{bibtex_page}{generate};

  die("no bibtex pages defined") unless $site->config->{bibtex_page}{pages};

  for my $p (keys %{$site->config->{bibtex_page}{pages}})
  {

    my $input_file = $site->config->{bibtex_page}{pages}{$p}{layout}
      or die "Must define bibtex_page.layout in config if bibtex_page.generate is enabled";

    my $bibtex_file = $site->config->{bibtex_page}{pages}{$p}{bibtex}
        or die "Must define bibtex_page.bibtex in config if bibtex_page.generate is enabled";

    if (! -e $site->config->{bibtex_page}{pages}{$p}{bibtex}) {
      die "bibtex_page.bibtex:".$site->config->{bibtex_page}{bibtex}." file must exist\n";
    }

    my $url = $site->config->{bibtex_page}{pages}{$p}{url} // 'publications/';

    my $destination = $site->config->{bibtex_page}{pages}{$p}{destination} // $site->destination;

    $self->_create_destination_directory_if_needed( $destination );

    # here we need to parse the bibtex file and generate the list
    my @publications;
    my $fh = IO::File->new($site->config->{bibtex_page}{pages}{$p}{bibtex});
    my $bibfile = BibTeX::Parser->new($fh);

    my @entries;
    while (my $entry = $bibfile->next ) {
      if ($entry->parse_ok) {
        push(@entries, $entry);
      }


    }


    # sort publications by year
    @publications=sort { $b->{year} <=> $a->{year} } @entries;

    # create the new page
    my $page = HiD::Page->new({
      dest_dir       => $destination ,
      hid            => $site ,
      url            => $url ,
      input_filename => $input_file ,
      layouts        => $site->layouts ,
    });
    $page->metadata->{publications} = \@publications;

    $site->add_input( "Publication" => 'page' );
    $site->add_object( $page );

    $site->INFO( "* Injected Bibtex page $p");
  }
}

__PACKAGE__->meta->make_immutable;
1;

__END__

=pod

=encoding UTF-8

=head1 NAME

HiD::Generator::BibtexPage - HiD Bibtex publication list page generator

=head1 VERSION

version 0.1

=head1 DESCRIPTION

This Generator produces a list of publications from a bibtex file.

Enable it by setting the 'bibtex_page.generate' key in your config to true
and the 'bibtex_page.layout' key to the path with the layout for the archive
page. You can also set 'bibtex_page.url' to the URL where the page should be
published to, or let it default to the site-wide destination. Finally,
'bibtex_page.destination' can be used to set a destination directory.

=head1 METHODS

=head2 generate

  reads the configured bibtex files and generates an html version

=head1 NAME

HiD::Generator::BibtexPage - Bibtex publication list page generator

=head1 AUTHOR

Dominik Meyer <dmeyer@federationhq.de>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by Dominik Meyer.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
