package PDK::Connector::Radware;

use utf8;
use v5.30;
use Moose;
use Carp;
use Data::Dumper;

# 继承连接器常用功能
with 'PDK::Connector::Role';

# 登录成功提示符（Radware设备特有格式）
has prompt => (
    is       => 'ro',
    required => 0,
    default  => '^>>.*?#\s*$', # 匹配Radware设备的提示符格式
);

# 特权模式状态标志（Radware设备默认进入特权模式）
has _enable_ => (
    is       => 'ro',
    required => 0,
    default  => 1, # Radware设备默认进入特权模式
);

# 等待设备响应，处理各种交互提示
sub waitfor {
    my ($self, $prompt) = @_;
    $prompt = $prompt // $self->{$prompt}; # 使用提供的提示符或默认提示符

    my $exp = $self->{exp};
    my $buff = "";

    # 等待设备响应，处理各种交互提示
    my @ret = $exp->expect($self->{timeout},
        # 处理保存确认提示
        [ qr/Confirm saving without first applying changes/i => sub {
            $self->send("y\r"); # 发送y确认
            $buff .= $exp->before() . $exp->match();
            exp_continue;
        } ],
        # 处理FLASH保存确认提示
        [ qr/Confirm saving to FLASH/i => sub {
            $self->send("y\r"); # 发送y确认
            $buff .= $exp->before() . $exp->match();
            exp_continue;
        } ],
        # 处理信息转储确认提示
        [ qr/Confirm dumping all information/i => sub {
            $self->send("y\r"); # 发送y确认
            $buff .= $exp->before() . $exp->match();
            exp_continue;
        } ],
        # 处理私钥显示确认提示
        [ qr/(Display|Include) private keys/i => sub {
            $self->send($self->{passphrase} ? "y\r" : "n\r"); # 根据是否有密码短语选择
            $buff .= $exp->before() . $exp->match();
            exp_continue;
        } ],
        # 处理密码短语输入提示
        [ qr/(Enter|Reconfirm) passphrase/i => sub {
            $self->send("$self->{passphrase}\r"); # 发送密码短语
            $buff .= $exp->before() . $exp->match();
            exp_continue;
        } ],
        # 匹配命令提示符
        [ qr/$prompt/m => sub {
            $buff .= $exp->before() . $exp->match(); # 保存完整输出
        } ]
    );

    # 检查Expect操作是否出错
    if (defined $ret[1]) {
        croak("[等待回显捕捉异常]" . $ret[3] . $ret[1]);
    }

    # 清理输出内容（Radware设备特有控制字符）
    $buff =~ s/\c@\cH+\s+\cH+//g; # 移除退格控制字符及其影响的文本
    $buff =~ s/\cM//g;            # 移除回车符（CR），保留换行符（LF）

    return $buff;
}

# 执行多个命令
sub execCommands {
    my ($self, $commands) = @_;

    # 检查登录状态，如果未登录则先登录
    if (!$self->{_login_}) {
        my $ret = $self->login();
        return $ret unless $ret->{success};
    }

    my $result = '';
    # 逐个执行命令
    for my $cmd (@{$commands}) {
        $self->send("$cmd\n");

        # 检查命令是否执行失败（Radware特有错误消息）
        my $buff = $self->waitfor();
        if ($buff =~ /^Error:.*?$/i) {
            return {
                success     => 0,
                failCommand => $cmd,
                result      => $result,
                reason      => $result . $buff
            };
        }
        else {
            $result .= $buff; # 保存命令执行结果
        }
    }

    return { success => 1, result => $result };
}

# 获取设备配置方法（Radware特有命令）
sub getConfig {
    my $self = shift;
    my $commands = [
        "cfg/dump", # 转储配置
        "cd",       # 切换目录（可能用于重置上下文）
    ];
    my $config = $self->execCommands($commands);
    my $lines = "";

    if ($config->{success} == 1) {
        $lines = $config->{result};
    }
    else {
        return $config; # 返回错误信息
    }

    return { success => 1, config => $lines };
}

# 使Moose类不可变
__PACKAGE__->meta->make_immutable;
1;
