# Development Notes for App::optex::up

## Getopt::EX::Config Integration

### Processing Order in Getopt::EX::Module

When loading a module, Getopt::EX::Module processes in this order:

1. `use Module` - Perl code executes
2. `readrc` - `__DATA__` section is processed, macros and `$ENV{...}` are expanded
3. `initialize` is called
4. `finalize` is called

**Important**: Environment variables set in `initialize`/`finalize` will NOT be reflected in `__DATA__` macro expansion because it happens earlier.

### Solution: Use `$mod->setopt` in finalize

Instead of relying on `__DATA__` macros, use `$mod->setopt(default => ...)` in `finalize` to dynamically set the default option. This overrides any existing definition because `setopt` uses `unshift` and `getopt` uses `first`.

### Getopt::EX::Config Key Names

Option names with hyphens must match exactly:

```perl
my $config = Getopt::EX::Config->new(
    'pane-width' => 85,  # Use hyphen, not underscore
);

# Access with the same key
my $width = $config->{'pane-width'};
```

### Terminal Width Detection

Use `Term::ReadKey` with `/dev/tty` for reliable terminal width:

```perl
use Term::ReadKey;

sub term_width {
    my @size;
    if (open my $tty, ">", "/dev/tty") {
        @size = GetTerminalSize $tty, $tty;
    }
    $size[0];
}
```

### Short Options

`Getopt::EX::Config` 1.00+ no longer uses `-C` for `--config`, so modules can safely use `-C` for their own options.

## Testing

```bash
# Test with development version
perl -Ilib -S optex -Mdebug -Mup date

# Test with options
perl -Ilib -S optex -Mdebug -Mup -C2 -- date
perl -Ilib -S optex -Mdebug -Mup --pane-width=40 -- date
```

## Build and Release

This project uses `minil` for build and release.

### Before Committing

**Important**: `README.md` is auto-generated by `minil` from POD in `lib/App/optex/up.pm`. Do not edit `README.md` directly.

1. Edit POD in `lib/App/optex/up.pm`
2. Run `minil build` to regenerate `README.md`
3. Commit both files

```bash
# After editing POD
minil build
git add lib/App/optex/up.pm README.md
git commit -m "..."
git push
```

### Release

```bash
minil release
```

This will:
- Confirm version number
- Run tests
- Upload to CPAN
- Create git tag

## Related Projects

### App-nup (submodule)

`App-nup` is a wrapper script for `optex -Mup`. When modifying `up.pm`:

- Check if the same changes are needed in `App-nup/script/nup`
- `App-nup/lib/App/nup.pm` is auto-generated from `script/nup`
- Keep option documentation order consistent between both projects
- Update `App-nup/cpanfile` if new `App::optex::up` version is required

## Dependencies

- Getopt::EX::Config >= 1.00
- Term::ReadKey
- App::ansicolumn
