/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.throttling;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelContextAware;
import org.apache.camel.Exchange;
import org.apache.camel.LoggingLevel;
import org.apache.camel.Route;
import org.apache.camel.RouteAware;
import org.apache.camel.spi.CamelLogger;
import org.apache.camel.spi.Configurer;
import org.apache.camel.spi.Metadata;
import org.apache.camel.support.RoutePolicySupport;
import org.apache.camel.throttling.ThrottlingExceptionHalfOpenHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Metadata(label="bean", description="A throttle based RoutePolicy which is modelled after the circuit breaker and will stop consuming from an endpoint based on the type of exceptions that are thrown and the threshold settings.", annotations={"interfaceName=org.apache.camel.spi.RoutePolicy"})
@Configurer(metadataOnly=true)
public class ThrottlingExceptionRoutePolicy
extends RoutePolicySupport
implements CamelContextAware,
RouteAware {
    private static final Logger LOG = LoggerFactory.getLogger(ThrottlingExceptionRoutePolicy.class);
    private static final int STATE_CLOSED = 0;
    private static final int STATE_HALF_OPEN = 1;
    private static final int STATE_OPEN = 2;
    private CamelContext camelContext;
    private Route route;
    private final Lock lock = new ReentrantLock();
    private CamelLogger stateLogger;
    @Metadata(description="How many failed messages within the window would trigger the circuit breaker to open", defaultValue="50")
    private int failureThreshold = 50;
    @Metadata(description="Sliding window for how long time to go back (in millis) when counting number of failures", defaultValue="60000")
    private long failureWindow = 60000L;
    @Metadata(description="Interval (in millis) for how often to check whether a currently open circuit breaker may work again", defaultValue="30000")
    private long halfOpenAfter = 30000L;
    @Metadata(description="Whether to always keep the circuit breaker open (never closes). This is only intended for development and testing purposes.")
    private boolean keepOpen;
    @Metadata(description="Allows to only throttle based on certain types of exceptions. Multiple exceptions (use FQN class name) can be separated by comma.")
    private String exceptions;
    @Metadata(description="Logging level for state changes", defaultValue="DEBUG")
    private LoggingLevel stateLoggingLevel = LoggingLevel.DEBUG;
    private List<Class<?>> throttledExceptions;
    @Metadata(label="advanced", description="Custom check to perform whether the circuit breaker can move to half-open state. If set then this is used instead of resuming the route.")
    private ThrottlingExceptionHalfOpenHandler halfOpenHandler;
    private final AtomicInteger failures = new AtomicInteger();
    private final AtomicInteger success = new AtomicInteger();
    private final AtomicInteger state = new AtomicInteger(0);
    private final AtomicBoolean keepOpenBool = new AtomicBoolean();
    private volatile Timer halfOpenTimer;
    private volatile long lastFailure;
    private volatile long openedAt;

    public ThrottlingExceptionRoutePolicy() {
    }

    public ThrottlingExceptionRoutePolicy(int threshold, long failureWindow, long halfOpenAfter, List<Class<?>> handledExceptions) {
        this(threshold, failureWindow, halfOpenAfter, handledExceptions, false);
    }

    public ThrottlingExceptionRoutePolicy(int threshold, long failureWindow, long halfOpenAfter, List<Class<?>> handledExceptions, boolean keepOpen) {
        this.throttledExceptions = handledExceptions;
        this.failureWindow = failureWindow;
        this.halfOpenAfter = halfOpenAfter;
        this.failureThreshold = threshold;
        this.keepOpenBool.set(keepOpen);
    }

    public void setCamelContext(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    public CamelContext getCamelContext() {
        return this.camelContext;
    }

    public Route getRoute() {
        return this.route;
    }

    public void setRoute(Route route) {
        this.route = route;
    }

    public List<Class<?>> getThrottledExceptions() {
        return this.throttledExceptions;
    }

    public String getExceptions() {
        return this.exceptions;
    }

    public void setExceptions(String exceptions) {
        this.exceptions = exceptions;
    }

    protected void doInit() throws Exception {
        super.doInit();
        this.stateLogger = new CamelLogger(LOG, this.stateLoggingLevel);
        if (this.exceptions != null && this.throttledExceptions == null) {
            ArrayList list = new ArrayList();
            for (String fqn : this.exceptions.split(",")) {
                Class clazz = this.camelContext.getClassResolver().resolveMandatoryClass(fqn);
                list.add(clazz);
            }
            this.throttledExceptions = list;
        }
    }

    @Override
    public void onInit(Route route) {
        LOG.debug("Initializing ThrottlingExceptionRoutePolicy route policy");
        this.logState();
    }

    @Override
    public void onStart(Route route) {
        if (this.keepOpenBool.get()) {
            this.openCircuit(route);
        }
    }

    protected void doStop() throws Exception {
        Timer timer = this.halfOpenTimer;
        if (timer != null) {
            timer.cancel();
            this.halfOpenTimer = null;
        }
    }

    @Override
    public void onExchangeDone(Route route, Exchange exchange) {
        if (this.keepOpenBool.get()) {
            if (this.state.get() != 2) {
                LOG.debug("Opening circuit (keepOpen is true)");
                this.openCircuit(route);
            }
        } else {
            if (this.hasFailed(exchange)) {
                this.failures.incrementAndGet();
                this.lastFailure = System.currentTimeMillis();
            } else {
                this.success.incrementAndGet();
            }
            this.calculateState(route);
        }
    }

    private boolean hasFailed(Exchange exchange) {
        if (exchange == null) {
            return false;
        }
        boolean answer = false;
        if (exchange.getException() != null) {
            if (this.throttledExceptions == null || this.throttledExceptions.isEmpty()) {
                answer = true;
            } else {
                for (Class<?> exception : this.throttledExceptions) {
                    if (exchange.getException(exception) == null) continue;
                    answer = true;
                    break;
                }
            }
        }
        if (LOG.isDebugEnabled()) {
            String exceptionName = exchange.getException() == null ? "none" : exchange.getException().getClass().getSimpleName();
            LOG.debug("hasFailed ({}) with Throttled Exception: {} for exchangeId: {}", new Object[]{answer, exceptionName, exchange.getExchangeId()});
        }
        return answer;
    }

    private void calculateState(Route route) {
        boolean failureLimitReached = this.isThresholdExceeded();
        if (this.state.get() == 0) {
            if (failureLimitReached) {
                LOG.debug("Opening circuit...");
                this.openCircuit(route);
            }
        } else if (this.state.get() == 1) {
            if (failureLimitReached) {
                LOG.debug("Opening circuit...");
                this.openCircuit(route);
            } else {
                LOG.debug("Closing circuit...");
                this.closeCircuit(route);
            }
        } else if (this.state.get() == 2) {
            if (!this.keepOpenBool.get()) {
                long elapsedTimeSinceOpened = System.currentTimeMillis() - this.openedAt;
                if (this.halfOpenAfter <= elapsedTimeSinceOpened) {
                    LOG.debug("Checking an open circuit...");
                    if (this.halfOpenHandler != null) {
                        if (this.halfOpenHandler.isReadyToBeClosed()) {
                            LOG.debug("Closing circuit...");
                            this.closeCircuit(route);
                        } else {
                            LOG.debug("Opening circuit...");
                            this.openCircuit(route);
                        }
                    } else {
                        LOG.debug("Half opening circuit...");
                        this.halfOpenCircuit(route);
                    }
                } else {
                    LOG.debug("Keeping circuit open (time not elapsed)...");
                }
            } else {
                LOG.debug("Keeping circuit open (keepOpen is true)...");
                this.addHalfOpenTimer(route);
            }
        }
    }

    protected boolean isThresholdExceeded() {
        boolean output = false;
        this.logState();
        if (this.failures.get() >= this.failureThreshold && this.lastFailure >= System.currentTimeMillis() - this.failureWindow) {
            output = true;
        }
        return output;
    }

    protected void openCircuit(Route route) {
        try {
            this.lock.lock();
            this.suspendOrStopConsumer(route.getConsumer());
            this.state.set(2);
            this.openedAt = System.currentTimeMillis();
            this.addHalfOpenTimer(route);
            this.logState();
        }
        catch (Exception e) {
            this.handleException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    protected void addHalfOpenTimer(Route route) {
        this.halfOpenTimer = new Timer();
        this.halfOpenTimer.schedule((TimerTask)new HalfOpenTask(route), this.halfOpenAfter);
    }

    protected void halfOpenCircuit(Route route) {
        try {
            this.lock.lock();
            this.resumeOrStartConsumer(route.getConsumer());
            this.state.set(1);
            this.logState();
        }
        catch (Exception e) {
            this.handleException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    protected void closeCircuit(Route route) {
        try {
            this.lock.lock();
            this.resumeOrStartConsumer(route.getConsumer());
            this.failures.set(0);
            this.success.set(0);
            this.lastFailure = 0L;
            this.openedAt = 0L;
            this.state.set(0);
            this.logState();
        }
        catch (Exception e) {
            this.handleException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void logState() {
        if (this.stateLogger != null) {
            this.stateLogger.log(this.dumpState());
        }
    }

    public String getStateAsString() {
        return ThrottlingExceptionRoutePolicy.stateAsString(this.state.get());
    }

    public String dumpState() {
        String routeState = this.getStateAsString();
        if (this.failures.get() > 0) {
            return String.format("State %s, failures %d, last failure %d ms ago", routeState, this.failures.get(), System.currentTimeMillis() - this.lastFailure);
        }
        return String.format("State %s, failures %d", routeState, this.failures.get());
    }

    private static String stateAsString(int num) {
        if (num == 0) {
            return "closed";
        }
        if (num == 1) {
            return "half opened";
        }
        return "opened";
    }

    public ThrottlingExceptionHalfOpenHandler getHalfOpenHandler() {
        return this.halfOpenHandler;
    }

    public void setHalfOpenHandler(ThrottlingExceptionHalfOpenHandler halfOpenHandler) {
        this.halfOpenHandler = halfOpenHandler;
    }

    public boolean getKeepOpen() {
        return this.keepOpenBool.get();
    }

    public void setKeepOpen(boolean keepOpen) {
        this.keepOpenBool.set(keepOpen);
    }

    public int getFailureThreshold() {
        return this.failureThreshold;
    }

    public void setFailureThreshold(int failureThreshold) {
        this.failureThreshold = failureThreshold;
    }

    public long getFailureWindow() {
        return this.failureWindow;
    }

    public void setFailureWindow(long failureWindow) {
        this.failureWindow = failureWindow;
    }

    public long getHalfOpenAfter() {
        return this.halfOpenAfter;
    }

    public void setHalfOpenAfter(long halfOpenAfter) {
        this.halfOpenAfter = halfOpenAfter;
    }

    public int getFailures() {
        return this.failures.get();
    }

    public int getSuccess() {
        return this.success.get();
    }

    public long getLastFailure() {
        return this.lastFailure;
    }

    public long getOpenedAt() {
        return this.openedAt;
    }

    public LoggingLevel getStateLoggingLevel() {
        return this.stateLoggingLevel;
    }

    public void setStateLoggingLevel(LoggingLevel stateLoggingLevel) {
        this.stateLoggingLevel = stateLoggingLevel;
        if (this.stateLogger != null) {
            this.stateLogger.setLevel(stateLoggingLevel);
        }
    }

    public void setStateLoggingLevel(String stateLoggingLevel) {
        this.setStateLoggingLevel(LoggingLevel.valueOf((String)stateLoggingLevel));
    }

    class HalfOpenTask
    extends TimerTask {
        private final Route route;

        HalfOpenTask(Route route) {
            this.route = route;
        }

        @Override
        public void run() {
            if (ThrottlingExceptionRoutePolicy.this.halfOpenTimer != null) {
                ThrottlingExceptionRoutePolicy.this.halfOpenTimer.cancel();
            }
            ThrottlingExceptionRoutePolicy.this.calculateState(this.route);
        }
    }
}

