/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hbase;

import java.io.IOException;
import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.TableInfoMissingException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hbase.HBCKFsUtils;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.com.google.common.primitives.Ints;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class HBCKFsTableDescriptors {
    private static final Logger LOG = LoggerFactory.getLogger(HBCKFsTableDescriptors.class);
    private final FileSystem fs;
    private final Path rootdir;
    static final String TABLEINFO_FILE_PREFIX = ".tableinfo";
    static final String TABLEINFO_DIR = ".tabledesc";
    static final String TMP_DIR = ".tmp";
    @VisibleForTesting
    static final Comparator<FileStatus> TABLEINFO_FILESTATUS_COMPARATOR = new Comparator<FileStatus>(){

        @Override
        public int compare(FileStatus left, FileStatus right) {
            return right.compareTo(left);
        }
    };
    private static final PathFilter TABLEINFO_PATHFILTER = new PathFilter(){

        public boolean accept(Path p) {
            return p.getName().startsWith(HBCKFsTableDescriptors.TABLEINFO_FILE_PREFIX);
        }
    };
    @VisibleForTesting
    static final int WIDTH_OF_SEQUENCE_ID = 10;
    private static final Pattern TABLEINFO_FILE_REGEX = Pattern.compile(".tableinfo(\\.([0-9]{10}))?(\\.([0-9]+))?$");

    public HBCKFsTableDescriptors(FileSystem fs, Path rootdir) {
        this.fs = fs;
        this.rootdir = rootdir;
    }

    public static FileStatus getTableInfoPath(FileSystem fs, Path tableDir) throws IOException {
        return HBCKFsTableDescriptors.getTableInfoPath(fs, tableDir, false);
    }

    private static FileStatus getTableInfoPath(FileSystem fs, Path tableDir, boolean removeOldFiles) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        return HBCKFsTableDescriptors.getCurrentTableInfoStatus(fs, tableInfoDir, removeOldFiles);
    }

    static FileStatus getCurrentTableInfoStatus(FileSystem fs, Path dir, boolean removeOldFiles) throws IOException {
        FileStatus[] status = HBCKFsUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER);
        if (status == null || status.length < 1) {
            return null;
        }
        FileStatus mostCurrent = null;
        for (FileStatus file : status) {
            if (mostCurrent != null && TABLEINFO_FILESTATUS_COMPARATOR.compare(file, mostCurrent) >= 0) continue;
            mostCurrent = file;
        }
        if (removeOldFiles && status.length > 1) {
            for (FileStatus file : status) {
                Path path = file.getPath();
                if (file.equals((Object)mostCurrent)) continue;
                if (!fs.delete(file.getPath(), false)) {
                    LOG.warn("Failed cleanup of " + path);
                    continue;
                }
                LOG.debug("Cleaned up old tableinfo file " + path);
            }
        }
        return mostCurrent;
    }

    @VisibleForTesting
    Path getTableDir(TableName tableName) {
        return HBCKFsUtils.getTableDir(this.rootdir, tableName);
    }

    private static String formatTableInfoSequenceId(int number) {
        byte[] b = new byte[10];
        int d = Math.abs(number);
        for (int i = b.length - 1; i >= 0; --i) {
            b[i] = (byte)(d % 10 + 48);
            d /= 10;
        }
        return Bytes.toString((byte[])b);
    }

    @VisibleForTesting
    static int getTableInfoSequenceId(Path p) {
        if (p == null) {
            return 0;
        }
        Matcher m = TABLEINFO_FILE_REGEX.matcher(p.getName());
        if (!m.matches()) {
            throw new IllegalArgumentException(p.toString());
        }
        String suffix = m.group(2);
        if (suffix == null || suffix.length() <= 0) {
            return 0;
        }
        return Integer.parseInt(m.group(2));
    }

    @VisibleForTesting
    static String getTableInfoFileName(int sequenceid) {
        return ".tableinfo." + HBCKFsTableDescriptors.formatTableInfoSequenceId(sequenceid);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        Path tableDir = HBCKFsUtils.getTableDir(hbaseRootDir, tableName);
        return HBCKFsTableDescriptors.getTableDescriptorFromFs(fs, tableDir);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir) throws IOException {
        FileStatus status = HBCKFsTableDescriptors.getTableInfoPath(fs, tableDir, false);
        if (status == null) {
            throw new TableInfoMissingException("No table descriptor file under " + tableDir);
        }
        return HBCKFsTableDescriptors.readTableDescriptor(fs, status);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static TableDescriptor readTableDescriptor(FileSystem fs, FileStatus status) throws IOException {
        int len = Ints.checkedCast((long)status.getLen());
        byte[] content = new byte[len];
        try (FSDataInputStream fsDataInputStream = fs.open(status.getPath());){
            fsDataInputStream.readFully(content);
        }
        TableDescriptor htd = null;
        try {
            htd = TableDescriptorBuilder.parseFrom((byte[])content);
        }
        catch (DeserializationException e) {
            throw new IOException("content=" + Bytes.toShort((byte[])content), e);
        }
        return htd;
    }

    public void deleteTableDescriptorIfExists(TableName tableName) throws IOException {
        Path tableDir = this.getTableDir(tableName);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        HBCKFsTableDescriptors.deleteTableDescriptorFiles(this.fs, tableInfoDir, Integer.MAX_VALUE);
    }

    private static void deleteTableDescriptorFiles(FileSystem fs, Path dir, int maxSequenceId) throws IOException {
        FileStatus[] status;
        for (FileStatus file : status = HBCKFsUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER)) {
            Path path = file.getPath();
            int sequenceId = HBCKFsTableDescriptors.getTableInfoSequenceId(path);
            if (sequenceId > maxSequenceId) continue;
            boolean success = HBCKFsUtils.delete(fs, path, false);
            if (success) {
                LOG.debug("Deleted " + path);
                continue;
            }
            LOG.error("Failed to delete table descriptor at " + path);
        }
    }

    private static Path writeTableDescriptor(FileSystem fs, TableDescriptor htd, Path tableDir, FileStatus currentDescriptorFile) throws IOException {
        int currentSequenceId;
        Path tmpTableDir = new Path(tableDir, TMP_DIR);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        int newSequenceId = currentSequenceId = currentDescriptorFile == null ? 0 : HBCKFsTableDescriptors.getTableInfoSequenceId(currentDescriptorFile.getPath());
        int retries = 10;
        int retrymax = currentSequenceId + retries;
        Path tableInfoDirPath = null;
        do {
            String filename;
            Path tempPath;
            if (fs.exists(tempPath = new Path(tmpTableDir, filename = HBCKFsTableDescriptors.getTableInfoFileName(++newSequenceId)))) {
                LOG.debug(tempPath + " exists; retrying up to " + retries + " times");
                continue;
            }
            tableInfoDirPath = new Path(tableInfoDir, filename);
            try {
                HBCKFsTableDescriptors.writeTD(fs, tempPath, htd);
                fs.mkdirs(tableInfoDirPath.getParent());
                if (!fs.rename(tempPath, tableInfoDirPath)) {
                    throw new IOException("Failed rename of " + tempPath + " to " + tableInfoDirPath);
                }
                LOG.debug("Wrote into " + tableInfoDirPath);
                break;
            }
            catch (IOException ioe) {
                if (ioe instanceof AccessControlException) {
                    throw ioe;
                }
                LOG.debug("Failed write and/or rename; retrying", ioe);
                if (!HBCKFsUtils.deleteDirectory(fs, tempPath)) {
                    LOG.warn("Failed cleanup of " + tempPath);
                }
                tableInfoDirPath = null;
            }
        } while (newSequenceId < retrymax);
        if (tableInfoDirPath != null) {
            HBCKFsTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, newSequenceId - 1);
        }
        return tableInfoDirPath;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeTD(FileSystem fs, Path p, TableDescriptor htd) throws IOException {
        try (FSDataOutputStream out = fs.create(p, false);){
            out.write(TableDescriptorBuilder.toByteArray((TableDescriptor)htd));
        }
    }

    public boolean createTableDescriptor(TableDescriptor htd, boolean forceCreation) throws IOException {
        Path tableDir = this.getTableDir(htd.getTableName());
        return this.createTableDescriptorForTableDirectory(tableDir, htd, forceCreation);
    }

    public boolean createTableDescriptorForTableDirectory(Path tableDir, TableDescriptor htd, boolean forceCreation) throws IOException {
        Path p;
        FileStatus status = HBCKFsTableDescriptors.getTableInfoPath(this.fs, tableDir);
        if (status != null) {
            LOG.debug("Current path=" + status.getPath());
            if (!forceCreation && this.fs.exists(status.getPath()) && status.getLen() > 0L && HBCKFsTableDescriptors.readTableDescriptor(this.fs, status).equals(htd)) {
                LOG.trace("TableInfo already exists.. Skipping creation");
                return false;
            }
        }
        return (p = HBCKFsTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, status)) != null;
    }
}

