/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <memory>
#include <optional>

#include <nanoarrow/nanoarrow.h>

#include "iceberg/iceberg_export.h"
#include "iceberg/result.h"
#include "iceberg/type_fwd.h"

namespace iceberg {

/// \brief Convert an Iceberg schema to an Arrow schema.
///
/// \param[in] schema The Iceberg schema to convert.
/// \param[out] out The Arrow schema to convert to.
/// \return An error if the conversion fails.
ICEBERG_EXPORT Status ToArrowSchema(const Schema& schema, ArrowSchema* out);

/// \brief Convert an Arrow schema to an Iceberg schema.
///
/// \param[in] schema The Arrow schema to convert.
/// \param[in] schema_id The schema ID of the Iceberg schema.
/// \return The Iceberg schema or an error if the conversion fails.
Result<std::unique_ptr<Schema>> FromArrowSchema(const ArrowSchema& schema,
                                                std::optional<int32_t> schema_id);

/// \brief Convert a struct type to an Iceberg schema.
///
/// \param[in] struct_type The struct type to convert.
/// \param[in] schema_id The schema ID of the Iceberg schema.
/// \return The Iceberg schema.
std::unique_ptr<Schema> FromStructType(StructType&& struct_type,
                                       std::optional<int32_t> schema_id);

std::unique_ptr<StructType> ToStructType(const Schema& schema);

}  // namespace iceberg
