/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.bifromq.basekv.localengine.rocksdb;

import com.google.protobuf.ByteString;
import java.util.Map;
import java.util.Optional;
import org.apache.bifromq.basekv.localengine.IKVSpaceIterator;
import org.apache.bifromq.basekv.localengine.metrics.KVSpaceOpMeters;
import org.apache.bifromq.basekv.proto.Boundary;
import org.slf4j.Logger;

class RocksDBKVSpaceCheckpointReader extends AbstractRocksDBKVSpaceReader {
    private final IRocksDBKVSpaceEpoch checkpoint;
    private final Map<ByteString, ByteString> metadata;
    // Hold a strong reference to prevent premature GC of checkpoint
    private RocksDBKVSpaceCheckpoint strongRef;

    RocksDBKVSpaceCheckpointReader(String id,
                                   KVSpaceOpMeters opMeters,
                                   Logger logger,
                                   RocksDBKVSpaceCheckpoint strongRef,
                                   IRocksDBKVSpaceEpoch checkpoint,
                                   Map<ByteString, ByteString> metadata) {
        super(id, opMeters, logger);
        // keep strong reference during reader lifetime
        this.strongRef = strongRef;
        this.checkpoint = checkpoint;
        this.metadata = metadata;
    }

    @Override
    protected IRocksDBKVSpaceEpoch handle() {
        return checkpoint;
    }

    @Override
    protected RocksDBSnapshot snapshot() {
        return new RocksDBSnapshot(this.handle(), null);
    }

    @Override
    protected Optional<ByteString> doMetadata(ByteString metaKey) {
        return Optional.ofNullable(metadata.get(metaKey));
    }

    @Override
    protected IKVSpaceIterator doNewIterator(Boundary subBoundary) {
        return new RocksDBKVSpaceIterator(snapshot(), subBoundary, new IteratorOptions(false, 52428));
    }

    @Override
    public void close() {
        // release strong reference to allow GC of checkpoint
        strongRef = null;
    }
}
