/*
 * (C) Copyright 2005- ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence Version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 *
 * In applying this licence, ECMWF does not waive the privileges and immunities granted to it by
 * virtue of its status as an intergovernmental organisation nor does it submit to any jurisdiction.
 */

#include <stdio.h>
#include <iostream>

#include "grib_api_internal.h"
#include "eccodes.h"

#define SIZE 1024 * 1024 * 20


static void deleter(void* data)
{
    delete[] static_cast<unsigned char*>(data);
}

static int test_message_ownership(const char* filename) // , size_t* previous_end_of_message)
{
    FILE* in = fopen(filename, "r");
    if (!in) {
        printf("Error opening file %s\n", filename);
        return 1;
    }

    while (true) {
        size_t buf_len = SIZE;
        void* buf = new unsigned char[SIZE];

        int err = wmo_read_any_from_file(in, buf, &buf_len);

        if (err == GRIB_END_OF_FILE && buf_len == 0) {
            break;
        }

        if (err != GRIB_SUCCESS) {
            printf("Error reading message at offset: %s\n", codes_get_error_message(err));
            return err;
        }

        codes_handle* h = codes_handle_new_from_message(NULL, buf, buf_len);
        if (!h) {
            printf("Error creating handle at offset\n");
            return 1;
        }

        err = codes_handle_change_buffer_ownership(h, deleter);

        if (err != GRIB_SUCCESS) {
            printf("Error taking message ownership at offset: %s\n", codes_get_error_message(err));
            codes_handle_delete(h);
            return err;
        }

        codes_handle_delete(h);
        // codes_context_free(NULL, buf); // Don't free the buffer here. Freed by codes_handle_delete
    }

    fclose(in);
    return 0;
}

static long fread_wrapper(void* stream_data, void* buffer, long len) {
    return fread(buffer, 1, len, (FILE*) stream_data);
}

static int test_codes_handle_from_stream(const char* filename)
{
    FILE* in = fopen(filename, "r");
    if (!in) {
        printf("Error opening file %s\n", filename);
        return 1;
    }

    codes_handle* h = NULL;
    int err = 0;

    while ((h = codes_handle_new_from_stream(NULL, reinterpret_cast<void*>(in), fread_wrapper, &err)) != NULL) {
        if (err != GRIB_SUCCESS) {
            printf("Error reading message from stream: %s\n", codes_get_error_message(err));
            return err;
        }

        long paramId = 0;
        err = codes_get_long(h, "paramId", &paramId); // Just a dummy operation
        if (err != GRIB_SUCCESS) {
            printf("Error getting paramId: %s\n", codes_get_error_message(err));
            codes_handle_delete(h);
            return err;
        }

        codes_handle_delete(h);
    }

    fclose(in);
    return 0;
}

size_t posInStream = 0;

static long fread_wrapper_with_offset(void* stream_data, void* buffer, long len) {
    size_t nBytesRead = fread(buffer, 1, len, (FILE*) stream_data);
    posInStream += nBytesRead;
    return nBytesRead;
}

static int test_codes_handle_from_stream_with_offset(const char* filename)
{
    std::cout << "Testing codes_handle_new_from_stream with offset..." << std::endl;
    FILE* in = fopen(filename, "r");
    if (!in) {
        printf("Error opening file %s\n", filename);
        return 1;
    }

    codes_handle* h = NULL;
    int err = 0;

    while ((h = codes_handle_new_from_stream(NULL, reinterpret_cast<void*>(in), fread_wrapper_with_offset, &err)) != NULL) {
        if (err != GRIB_SUCCESS) {
            printf("Error reading message from stream: %s\n", codes_get_error_message(err));
            return err;
        }

        size_t messageSize = 0;
        err = codes_get_message_size(h, &messageSize);
        if (err != GRIB_SUCCESS) {
            printf("Error getting message length: %s\n", codes_get_error_message(err));
            codes_handle_delete(h);
            return err;
        }

        std::cout << "Message offset: " << posInStream - messageSize << " size: " << messageSize << std::endl;

        long paramId = 0;
        err = codes_get_long(h, "paramId", &paramId); // Just a dummy operation
        if (err != GRIB_SUCCESS) {
            printf("Error getting paramId: %s\n", codes_get_error_message(err));
            codes_handle_delete(h);
            return err;
        }

        codes_handle_delete(h);
    }

    fclose(in);
    return 0;
}

int main(int argc, char** argv)
{
    int err = 0;

    if (argc != 2) return 1;

    err = test_message_ownership(argv[1]);
    if (err) return err;

    err = test_codes_handle_from_stream(argv[1]);
    if (err) return err;

    // TODO(maee): Check why this fails on Windows?
//#ifndef ECCODES_ON_WINDOWS
    err = test_codes_handle_from_stream_with_offset(argv[1]);
    if (err) return err;
//#endif

    return 0;
}
