/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.row.columnar;

import org.apache.fluss.row.BinaryString;
import org.apache.fluss.row.Decimal;
import org.apache.fluss.row.InternalArray;
import org.apache.fluss.row.InternalMap;
import org.apache.fluss.row.InternalRow;
import org.apache.fluss.row.TimestampLtz;
import org.apache.fluss.row.TimestampNtz;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Objects;

/** Columnar array to support access to vector column data. */
public final class ColumnarArray implements InternalArray, Serializable {

    private static final long serialVersionUID = 1L;

    private final ColumnVector data;
    private final int offset;
    private final int numElements;

    public ColumnarArray(ColumnVector data, int offset, int numElements) {
        this.data = data;
        this.offset = offset;
        this.numElements = numElements;
    }

    @Override
    public int size() {
        return numElements;
    }

    @Override
    public boolean isNullAt(int pos) {
        return data.isNullAt(offset + pos);
    }

    @Override
    public boolean getBoolean(int pos) {
        return ((BooleanColumnVector) data).getBoolean(offset + pos);
    }

    @Override
    public byte getByte(int pos) {
        return ((ByteColumnVector) data).getByte(offset + pos);
    }

    @Override
    public short getShort(int pos) {
        return ((ShortColumnVector) data).getShort(offset + pos);
    }

    @Override
    public int getInt(int pos) {
        return ((IntColumnVector) data).getInt(offset + pos);
    }

    @Override
    public long getLong(int pos) {
        return ((LongColumnVector) data).getLong(offset + pos);
    }

    @Override
    public float getFloat(int pos) {
        return ((FloatColumnVector) data).getFloat(offset + pos);
    }

    @Override
    public double getDouble(int pos) {
        return ((DoubleColumnVector) data).getDouble(offset + pos);
    }

    @Override
    public BinaryString getChar(int pos, int length) {
        BytesColumnVector.Bytes byteArray = getByteArray(pos);
        return BinaryString.fromBytes(byteArray.data, byteArray.offset, length);
    }

    @Override
    public BinaryString getString(int pos) {
        BytesColumnVector.Bytes byteArray = getByteArray(pos);
        return BinaryString.fromBytes(byteArray.data, byteArray.offset, byteArray.len);
    }

    @Override
    public Decimal getDecimal(int pos, int precision, int scale) {
        return ((DecimalColumnVector) data).getDecimal(offset + pos, precision, scale);
    }

    @Override
    public TimestampNtz getTimestampNtz(int pos, int precision) {
        return ((TimestampNtzColumnVector) data).getTimestampNtz(offset + pos, precision);
    }

    @Override
    public TimestampLtz getTimestampLtz(int pos, int precision) {
        return ((TimestampLtzColumnVector) data).getTimestampLtz(offset + pos, precision);
    }

    @Override
    public byte[] getBinary(int pos, int length) {
        return getBytes(pos);
    }

    @Override
    public byte[] getBytes(int pos) {
        BytesColumnVector.Bytes byteArray = getByteArray(pos);
        if (byteArray.len == byteArray.data.length) {
            return byteArray.data;
        } else {
            return Arrays.copyOfRange(
                    byteArray.data, byteArray.offset, byteArray.offset + byteArray.len);
        }
    }

    @Override
    public InternalArray getArray(int pos) {
        return ((ArrayColumnVector) data).getArray(offset + pos);
    }

    @Override
    public InternalMap getMap(int pos) {
        return ((MapColumnVector) data).getMap(offset + pos);
    }

    @Override
    public InternalRow getRow(int pos, int numFields) {
        return ((RowColumnVector) data).getRow(offset + pos);
    }

    @Override
    public boolean[] toBooleanArray() {
        boolean[] res = new boolean[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getBoolean(i);
        }
        return res;
    }

    @Override
    public byte[] toByteArray() {
        byte[] res = new byte[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getByte(i);
        }
        return res;
    }

    @Override
    public short[] toShortArray() {
        short[] res = new short[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getShort(i);
        }
        return res;
    }

    @Override
    public int[] toIntArray() {
        int[] res = new int[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getInt(i);
        }
        return res;
    }

    @Override
    public long[] toLongArray() {
        long[] res = new long[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getLong(i);
        }
        return res;
    }

    @Override
    public float[] toFloatArray() {
        float[] res = new float[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getFloat(i);
        }
        return res;
    }

    @Override
    public double[] toDoubleArray() {
        double[] res = new double[numElements];
        for (int i = 0; i < numElements; i++) {
            res[i] = getDouble(i);
        }
        return res;
    }

    private BytesColumnVector.Bytes getByteArray(int pos) {
        return ((BytesColumnVector) data).getBytes(offset + pos);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ColumnarArray)) {
            return false;
        }
        ColumnarArray that = (ColumnarArray) o;
        return offset == that.offset
                && numElements == that.numElements
                && Objects.equals(data, that.data);
    }

    @Override
    public int hashCode() {
        return Objects.hash(data, offset, numElements);
    }
}
