/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.base.Preconditions;
import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.server.ServerUtils;
import org.apache.hadoop.hdds.utils.db.DBColumnFamilyDefinition;
import org.apache.hadoop.hdds.utils.db.DBConfigFromFile;
import org.apache.hadoop.hdds.utils.db.DBDefinition;
import org.apache.hadoop.hdds.utils.db.DBProfile;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.RDBStore;
import org.apache.hadoop.hdds.utils.db.RocksDBConfiguration;
import org.apache.hadoop.hdds.utils.db.RocksDatabaseException;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.managed.ManagedColumnFamilyOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedDBOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedLogger;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksDB;
import org.apache.hadoop.hdds.utils.db.managed.ManagedStatistics;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.rocksdb.InfoLogLevel;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.Statistics;
import org.rocksdb.StatsLevel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DBStoreBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(DBStoreBuilder.class);
    public static final Logger ROCKS_DB_LOGGER = LoggerFactory.getLogger((Class)ManagedRocksDB.ORIGINAL_CLASS);
    public static final String DEFAULT_COLUMN_FAMILY_NAME = StringUtils.bytes2String((byte[])RocksDB.DEFAULT_COLUMN_FAMILY);
    public static final DBProfile HDDS_DEFAULT_DB_PROFILE = DBProfile.DISK;
    private ManagedDBOptions rocksDBOption;
    private ManagedColumnFamilyOptions defaultCfOptions;
    private ManagedStatistics statistics;
    private String dbname;
    private Path dbPath;
    private Path optionsPath;
    private String dbJmxBeanNameName;
    private Map<String, ManagedColumnFamilyOptions> cfOptions = new HashMap<String, ManagedColumnFamilyOptions>();
    private ConfigurationSource configuration;
    private String rocksDbStat;
    private long rocksDbCfWriteBufferSize;
    private RocksDBConfiguration rocksDBConfiguration;
    private boolean openReadOnly = false;
    private final DBProfile defaultCfProfile;
    private boolean enableCompactionDag;
    private boolean createCheckpointDirs = true;
    private boolean enableRocksDbMetrics = true;
    private long maxDbUpdatesSizeThreshold;
    private Integer maxNumberOfOpenFiles = null;

    public static DBStore createDBStore(ConfigurationSource configuration, DBDefinition definition) throws RocksDatabaseException {
        return DBStoreBuilder.newBuilder(configuration, definition, null, null).build();
    }

    public static DBStoreBuilder newBuilder(ConfigurationSource conf, DBDefinition definition, File dbDir) {
        return DBStoreBuilder.newBuilder(conf, definition, dbDir.getName(), dbDir.getParentFile().toPath());
    }

    public static DBStoreBuilder newBuilder(ConfigurationSource conf, DBDefinition definition, String name, Path metadataDir) {
        return DBStoreBuilder.newBuilder(conf).apply(definition, name, metadataDir);
    }

    public static DBStoreBuilder newBuilder(ConfigurationSource configuration) {
        return new DBStoreBuilder(configuration, (RocksDBConfiguration)configuration.getObject(RocksDBConfiguration.class));
    }

    private DBStoreBuilder(ConfigurationSource configuration, RocksDBConfiguration rocksDBConfiguration) {
        this.configuration = configuration;
        this.rocksDbStat = configuration.getTrimmed("ozone.metastore.rocksdb.statistics", "OFF");
        this.rocksDbCfWriteBufferSize = (long)configuration.getStorageSize("ozone.metastore.rocksdb.cf.write.buffer.size", "128MB", StorageUnit.BYTES);
        this.rocksDBConfiguration = rocksDBConfiguration;
        this.defaultCfProfile = (DBProfile)this.configuration.getEnum("hdds.db.profile", (Enum)HDDS_DEFAULT_DB_PROFILE);
        LOG.debug("Default DB profile:{}", (Object)this.defaultCfProfile);
        this.maxDbUpdatesSizeThreshold = (long)configuration.getStorageSize("ozone.om.delta.update.data.size.max.limit", "1024MB", StorageUnit.BYTES);
    }

    public static File getDBDirPath(DBDefinition definition, ConfigurationSource configuration) {
        File metadataDir = definition.getDBLocation(configuration);
        if (metadataDir == null) {
            LOG.warn("{} is not configured. We recommend adding this setting. Falling back to {} instead.", (Object)definition.getLocationConfigKey(), (Object)"ozone.metadata.dirs");
            metadataDir = ServerUtils.getOzoneMetaDirPath(configuration);
        }
        return metadataDir;
    }

    private DBStoreBuilder apply(DBDefinition definition, String name, Path metadataDir) {
        if (name == null) {
            name = definition.getName();
        }
        this.setName(name);
        if (metadataDir == null) {
            metadataDir = DBStoreBuilder.getDBDirPath(definition, this.configuration).toPath();
        }
        this.setPath(metadataDir);
        this.setOptionsPath(definition.getOptionsPath(this.configuration));
        for (DBColumnFamilyDefinition<?, ?> columnFamily : definition.getColumnFamilies()) {
            ManagedColumnFamilyOptions cfOptionsFromFile = this.getCfOptionsFromFile(columnFamily.getName());
            if (cfOptionsFromFile == null) {
                cfOptionsFromFile = columnFamily.getCfOptions();
            }
            this.addTable(columnFamily.getName(), cfOptionsFromFile);
        }
        return this;
    }

    private void setDBOptionsProps(ManagedDBOptions dbOptions) {
        if (this.maxNumberOfOpenFiles != null) {
            dbOptions.setMaxOpenFiles(this.maxNumberOfOpenFiles.intValue());
        }
        if (!this.rocksDbStat.equals("OFF")) {
            this.statistics = new ManagedStatistics();
            this.statistics.setStatsLevel(StatsLevel.valueOf((String)this.rocksDbStat));
            dbOptions.setStatistics((Statistics)this.statistics);
        }
    }

    public RDBStore build() throws RocksDatabaseException {
        if (org.apache.commons.lang3.StringUtils.isBlank((CharSequence)this.dbname) || this.dbPath == null) {
            LOG.error("Required Parameter missing.");
            throw new RocksDatabaseException("Required parameter is missing. Please make sure Path and DB name is provided.");
        }
        Set<TableConfig> tableConfigs = this.makeTableConfigs();
        ManagedWriteOptions writeOptions = null;
        try {
            if (this.rocksDBOption == null) {
                this.rocksDBOption = this.getDefaultDBOptions();
            }
            this.setDBOptionsProps(this.rocksDBOption);
            File dbFile = this.getDBFile();
            if (!dbFile.getParentFile().exists()) {
                throw new RocksDatabaseException("The DB destination directory should exist.");
            }
            writeOptions = new ManagedWriteOptions();
            writeOptions.setSync(this.rocksDBConfiguration.getSyncOption());
            RDBStore rDBStore = new RDBStore(dbFile, this.rocksDBOption, this.statistics, writeOptions, tableConfigs, this.openReadOnly, this.dbJmxBeanNameName, this.enableCompactionDag, this.maxDbUpdatesSizeThreshold, this.createCheckpointDirs, this.configuration, this.enableRocksDbMetrics);
            return rDBStore;
        }
        catch (Exception ex) {
            try {
                if (writeOptions != null) {
                    writeOptions.close();
                }
            }
            catch (Exception suppressed) {
                ex.addSuppressed(suppressed);
            }
            throw ex;
        }
        finally {
            tableConfigs.forEach(TableConfig::close);
        }
    }

    public DBStoreBuilder setName(String name) {
        this.dbname = name;
        return this;
    }

    public DBStoreBuilder setDBJmxBeanNameName(String name) {
        this.dbJmxBeanNameName = name;
        return this;
    }

    public DBStoreBuilder addTable(String tableName) {
        return this.addTable(tableName, null);
    }

    public DBStoreBuilder addTable(String tableName, ManagedColumnFamilyOptions options) {
        this.cfOptions.put(tableName, options);
        return this;
    }

    public DBStoreBuilder setDBOptions(ManagedDBOptions option) {
        this.rocksDBOption = option;
        return this;
    }

    public DBStoreBuilder setDefaultCFOptions(ManagedColumnFamilyOptions options) {
        this.defaultCfOptions = options;
        return this;
    }

    public DBStoreBuilder setPath(Path path) {
        Preconditions.checkNotNull((Object)path);
        this.dbPath = path;
        return this;
    }

    public DBStoreBuilder setOptionsPath(Path optionsPath) {
        Preconditions.checkNotNull((Object)optionsPath);
        this.optionsPath = optionsPath;
        return this;
    }

    public DBStoreBuilder setOpenReadOnly(boolean readOnly) {
        this.openReadOnly = readOnly;
        return this;
    }

    public DBStoreBuilder setEnableCompactionDag(boolean enableCompactionDag) {
        this.enableCompactionDag = enableCompactionDag;
        return this;
    }

    public DBStoreBuilder setCreateCheckpointDirs(boolean createCheckpointDirs) {
        this.createCheckpointDirs = createCheckpointDirs;
        return this;
    }

    public DBStoreBuilder setEnableRocksDbMetrics(boolean enableRocksDbMetrics) {
        this.enableRocksDbMetrics = enableRocksDbMetrics;
        return this;
    }

    public DBStoreBuilder setProfile(DBProfile prof) {
        this.setDBOptions(prof.getDBOptions());
        this.setDefaultCFOptions(prof.getColumnFamilyOptions());
        return this;
    }

    public DBStoreBuilder setMaxNumberOfOpenFiles(Integer maxNumberOfOpenFiles) {
        this.maxNumberOfOpenFiles = maxNumberOfOpenFiles;
        return this;
    }

    private Set<TableConfig> makeTableConfigs() {
        HashSet<TableConfig> tableConfigs = new HashSet<TableConfig>();
        ManagedColumnFamilyOptions usedCFOptions = this.getFromFileOrDefault(DEFAULT_COLUMN_FAMILY_NAME);
        usedCFOptions.setWriteBufferSize(this.rocksDbCfWriteBufferSize);
        this.cfOptions.putIfAbsent(DEFAULT_COLUMN_FAMILY_NAME, usedCFOptions);
        for (Map.Entry<String, ManagedColumnFamilyOptions> entry : this.cfOptions.entrySet()) {
            String name = entry.getKey();
            ManagedColumnFamilyOptions options = entry.getValue();
            if (options == null) {
                LOG.debug("using default column family options for table: {}", (Object)name);
                ManagedColumnFamilyOptions fromFileOrDefault = this.getFromFileOrDefault(name);
                fromFileOrDefault.setWriteBufferSize(this.rocksDbCfWriteBufferSize);
                tableConfigs.add(new TableConfig(name, fromFileOrDefault));
                continue;
            }
            tableConfigs.add(new TableConfig(name, options));
        }
        return tableConfigs;
    }

    private ManagedColumnFamilyOptions getDefaultCfOptions() {
        return Optional.ofNullable(this.defaultCfOptions).orElseGet(this.defaultCfProfile::getColumnFamilyOptions);
    }

    private ManagedColumnFamilyOptions getFromFileOrDefault(String cfName) {
        ManagedColumnFamilyOptions cfOptionsFromFile = this.getCfOptionsFromFile(cfName);
        return cfOptionsFromFile != null ? cfOptionsFromFile : this.getDefaultCfOptions();
    }

    public DBStoreBuilder disableDefaultCFAutoCompaction(boolean defaultCFAutoCompaction) {
        ManagedColumnFamilyOptions defaultCFOptions = this.getFromFileOrDefault(DEFAULT_COLUMN_FAMILY_NAME);
        defaultCFOptions.setDisableAutoCompactions(defaultCFAutoCompaction);
        this.setDefaultCFOptions(defaultCFOptions);
        return this;
    }

    private ManagedDBOptions getDefaultDBOptions() {
        ManagedDBOptions dbOptions = null;
        try {
            Path configuredPath = this.optionsPath != null ? this.optionsPath : this.dbPath;
            dbOptions = DBConfigFromFile.readDBOptionsFromFile(configuredPath);
        }
        catch (RocksDBException e) {
            LOG.error("Error trying to use dbOptions from file: {}", (Object)this.optionsPath);
        }
        if (dbOptions == null) {
            dbOptions = this.defaultCfProfile.getDBOptions();
        }
        if (this.rocksDBConfiguration.isRocksdbLoggingEnabled()) {
            ManagedLogger logger = new ManagedLogger(dbOptions, (infoLogLevel, s) -> ROCKS_DB_LOGGER.info(s));
            InfoLogLevel level = InfoLogLevel.valueOf((String)(this.rocksDBConfiguration.getRocksdbLogLevel() + "_LEVEL"));
            logger.setInfoLogLevel(level);
            dbOptions.setLogger((org.rocksdb.Logger)logger);
        }
        dbOptions.setMaxLogFileSize(this.rocksDBConfiguration.getMaxLogFileSize());
        dbOptions.setKeepLogFileNum((long)this.rocksDBConfiguration.getKeepLogFileNum());
        dbOptions.setWalTtlSeconds(this.rocksDBConfiguration.getWalTTL());
        dbOptions.setWalSizeLimitMB(this.rocksDBConfiguration.getWalSizeLimit());
        return dbOptions;
    }

    public ManagedColumnFamilyOptions getCfOptionsFromFile(String cfName) {
        if (Objects.isNull((Object)this.defaultCfProfile)) {
            throw new RuntimeException();
        }
        Path configuredPath = this.optionsPath != null ? this.optionsPath : this.dbPath;
        ManagedColumnFamilyOptions cfOptionsFromFile = null;
        try {
            cfOptionsFromFile = DBConfigFromFile.readCFOptionsFromFile(configuredPath, cfName);
        }
        catch (RocksDBException e) {
            LOG.error("Error while trying to read ColumnFamilyOptions from file: {}", (Object)configuredPath);
        }
        return cfOptionsFromFile;
    }

    private File getDBFile() throws RocksDatabaseException {
        if (this.dbPath == null) {
            LOG.error("DB path is required.");
            throw new RocksDatabaseException("A Path to for DB file is needed.");
        }
        if (org.apache.commons.lang3.StringUtils.isBlank((CharSequence)this.dbname)) {
            LOG.error("DBName is a required.");
            throw new RocksDatabaseException("A valid DB name is required.");
        }
        return Paths.get(this.dbPath.toString(), this.dbname).toFile();
    }
}

