package Venus::Role::Matchable;

use 5.018;

use strict;
use warnings;

# IMPORTS

use Venus::Role 'with';

# METHODS

sub match {
  my ($self, $method, @args) = @_;

  require Venus::Match;

  local $_ = $self;

  my $match = Venus::Match->new($method ? scalar($self->$method(@args)) : $self);

  return $match;
}

# EXPORTS

sub EXPORT {
  ['match']
}

1;



=head1 NAME

Venus::Role::Matchable - Matchable Role

=cut

=head1 ABSTRACT

Matchable Role for Perl 5

=cut

=head1 SYNOPSIS

  package Example;

  use Venus::Class;

  with 'Venus::Role::Matchable';

  attr 'active';

  sub validate {
    my ($self) = @_;

    return $self->match->when('active')->then(true)->none(false);
  }

  package main;

  my $example = Example->new;

  # $example->validate->result;

  # 0

=cut

=head1 DESCRIPTION

This package modifies the consuming package and provides a mechanism for
assembling complex pattern matching operations.

=cut

=head1 METHODS

This package provides the following methods:

=cut

=head2 match

  match(string | coderef $method, any @args) (Venus::Match)

The match method returns a L<Venus::Match> object having the match value set to
the invocant or the result of a dispatch. This method supports dispatching,
i.e. providing a method name and arguments whose return value will be acted on
by this method.

I<Since C<0.04>>

=over 4

=item match example 1

  package main;

  my $example = Example->new;

  my $match = $example->match;

  # bless({..., value => bless(..., 'Example')}, 'Venus::Match')

=back

=over 4

=item match example 2

  package main;

  my $example = Example->new;

  my $match = $example->match('active');

  # bless({..., value => undef}, 'Venus::Match')

=back

=over 4

=item match example 3

  package main;

  my $example = Example->new(active => 1);

  my $match = $example->match('active');

  # bless({..., value => 1}, 'Venus::Match')

=back

=cut

=head1 AUTHORS

Awncorp, C<awncorp@cpan.org>

=cut

=head1 LICENSE

Copyright (C) 2022, Awncorp, C<awncorp@cpan.org>.

This program is free software, you can redistribute it and/or modify it under
the terms of the Apache license version 2.0.

=cut