// Copyright (C) 2024 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import {AreaSelection} from '../../public/selection';
import {CPU_SLICE_TRACK_KIND} from '../../public/track_kinds';
import {Engine} from '../../trace_processor/engine';
import {exists} from '../../base/utils';
import {Aggregator} from '../../components/aggregation_adapter';
import {WattsonAggregationPanel} from './aggregation_panel';
import {ColumnDef, Sorting} from '../../components/aggregation';

export class WattsonPackageSelectionAggregator implements Aggregator {
  readonly id = 'wattson_plugin_package_aggregation';
  readonly PanelComponent = WattsonAggregationPanel;

  probe(area: AreaSelection) {
    const selectedCpus: number[] = [];
    for (const trackInfo of area.tracks) {
      if (trackInfo?.tags?.kinds?.includes(CPU_SLICE_TRACK_KIND)) {
        exists(trackInfo.tags.cpu) && selectedCpus.push(trackInfo.tags.cpu);
      }
    }
    if (selectedCpus.length === 0) return undefined;

    return {
      prepareData: async (engine: Engine) => {
        await engine.query(`drop view if exists ${this.id};`);

        const duration = area.end - area.start;

        // Prerequisite tables are already generated by Wattson thread aggregation,
        // which is run prior to execution of this module
        await engine.query(`
          INCLUDE PERFETTO MODULE wattson.estimates;
          INCLUDE PERFETTO MODULE wattson.tasks.idle_transitions_attribution;

          -- Group idle attribution by package
          CREATE OR REPLACE PERFETTO TABLE
          wattson_plugin_per_package_idle_attribution AS
          SELECT
            SUM(idle_cost_mws) as idle_cost_mws,
            uid
          FROM wattson_plugin_idle_attribution
          JOIN thread USING(utid)
          JOIN process USING(upid)
          GROUP BY uid;

          -- Grouped by UID and made CPU agnostic
          CREATE PERFETTO VIEW ${this.id} AS
          WITH base AS (
            SELECT
              ROUND(SUM(total_pws) / ${duration}, 3) as active_mw,
              ROUND(SUM(total_pws) / 1000000000, 3) as active_mws,
              ROUND(COALESCE(idle_cost_mws, 0), 3) as idle_cost_mws,
              ROUND(
                COALESCE(idle_cost_mws, 0) + SUM(total_pws) / 1000000000,
                3
              ) as total_mws,
              wattson_plugin_unioned_per_cpu_total.uid AS uid,
              package_name
            FROM wattson_plugin_unioned_per_cpu_total
            LEFT JOIN wattson_plugin_per_package_idle_attribution
              ON wattson_plugin_unioned_per_cpu_total.uid IS
              wattson_plugin_per_package_idle_attribution.uid
            GROUP BY wattson_plugin_unioned_per_cpu_total.uid, package_name
          )
          select *,
            total_mws / (SUM(total_mws) OVER()) AS percent_of_total_energy
            from base;
        `);

        return {
          tableName: this.id,
        };
      },
    };
  }

  getColumnDefinitions(): ColumnDef[] {
    return [
      {
        title: 'Package Name',
        columnId: 'package_name',
      },
      {
        title: 'Android app UID',
        columnId: 'uid',
      },
      {
        title: 'Active power (estimated mW)',
        columnId: 'active_mw',
        sum: true,
      },
      {
        title: 'Active energy (estimated mWs)',
        columnId: 'active_mws',
        sum: true,
      },
      {
        title: 'Idle transitions overhead (estimated mWs)',
        columnId: 'idle_cost_mws',
        sum: false,
      },
      {
        title: 'Total energy (estimated mWs)',
        columnId: 'total_mws',
        sum: true,
      },
      {
        title: '% of total energy',
        formatHint: 'PERCENT',
        columnId: 'percent_of_total_energy',
        sum: false,
      },
    ];
  }

  getTabName() {
    return 'Wattson by package';
  }

  getDefaultSorting(): Sorting {
    return {column: 'active_mws', direction: 'DESC'};
  }
}
