module Toys
  module Utils
    ##
    # **_Defined in the toys-core gem_**
    #
    # An object that implements standard UI elements, such as error reports and
    # logging, as provided by the `toys` command line. Specifically, it
    # implements pretty formatting of log entries and stack traces, and renders
    # using ANSI coloring where available via {Toys::Utils::Terminal}.
    #
    # This object can be used to implement `toys`-style behavior when creating
    # a CLI object. For example:
    #
    #     require "toys/utils/standard_ui"
    #     ui = Toys::Utils::StandardUI.new
    #     cli = Toys::CLI.new(**ui.cli_args)
    #
    class StandardUI
      ##
      # Create a Standard UI.
      #
      # By default, all output is written to `$stderr`, and will share a single
      # {Toys::Utils::Terminal} object, allowing multiple tools and/or threads
      # to interleave messages without interrupting one another.
      #
      # @param output [IO,Toys::Utils::Terminal] Where to write output. You can
      #     pass a terminal object, or an IO stream that will be wrapped in a
      #     terminal output. Default is `$stderr`.
      #
      def initialize(output: nil)
        # Source available in the toys-core gem
      end

      ##
      # The terminal underlying this UI
      #
      # @return [Toys::Utils::Terminal]
      #
      attr_reader :terminal

      ##
      # A hash that maps severities to styles recognized by
      # {Toys::Utils::Terminal}. Used to style the header for each log entry.
      # This hash can be modified in place to adjust the behavior of loggers
      # created by this UI.
      #
      # @return [Hash{String => Array<Symbol>}]
      #
      attr_reader :log_header_severity_styles

      ##
      # Convenience method that returns a hash of arguments that can be passed
      # to the {Toys::CLI} constructor. Includes the `:error_handler` and
      # `:logger_factory` arguments.
      #
      # @return [Hash]
      #
      def cli_args
        # Source available in the toys-core gem
      end

      ##
      # Returns an error handler conforming to the `:error_handler` argument to
      # the {Toys::CLI} constructor. Specifically, it returns the
      # {#error_handler_impl} method as a proc.
      #
      # @return [Proc]
      #
      def error_handler
        # Source available in the toys-core gem
      end

      ##
      # Returns a logger factory conforming to the `:logger_factory` argument
      # to the {Toys::CLI} constructor. Specifically, it returns the
      # {#logger_factory_impl} method as a proc.
      #
      # @return [Proc]
      #
      def logger_factory
        # Source available in the toys-core gem
      end

      ##
      # Implementation of the error handler. As dictated by the error handler
      # specification in {Toys::CLI}, this must take a {Toys::ContextualError}
      # as an argument, and return an exit code.
      #
      # The base implementation uses {#display_error_notice} and
      # {#display_signal_notice} to print an appropriate message to the UI's
      # terminal, and uses {#exit_code_for} to determine the correct exit code.
      # Any of those methods can be overridden by a subclass to alter their
      # behavior, or this main implementation method can be overridden to
      # change the overall behavior.
      #
      # @param error [Toys::ContextualError] The error received
      # @return [Integer] The exit code
      #
      def error_handler_impl(error)
        # Source available in the toys-core gem
      end

      ##
      # Implementation of the logger factory. As dictated by the logger factory
      # specification in {Toys::CLI}, this must take a {Toys::ToolDefinition}
      # as an argument, and return a `Logger`.
      #
      # The base implementation returns a logger that writes to the UI's
      # terminal, using {#logger_formatter_impl} as the formatter. It sets the
      # level to `Logger::WARN` by default. Either this method or the helper
      # methods can be overridden to change this behavior.
      #
      # @param _tool {Toys::ToolDefinition} The tool definition of the tool to
      #     be executed
      # @return [Logger]
      #
      def logger_factory_impl(_tool)
        # Source available in the toys-core gem
      end

      ##
      # Returns an exit code appropriate for the given exception. Currently,
      # the logic interprets signals (returning the convention of 128 + signo),
      # usage errors (returning the conventional value of 2), and tool not
      # runnable errors (returning the conventional value of 126), and defaults
      # to 1 for all other error types.
      #
      # This method is used by {#error_handler_impl} and can be overridden to
      # change its behavior.
      #
      # @param error [Exception] The exception raised. This method expects the
      #     original exception, rather than a ContextualError.
      # @return [Integer] The appropriate exit code
      #
      def exit_code_for(error)
        # Source available in the toys-core gem
      end

      ##
      # Displays a default output for a signal received.
      #
      # This method is used by {#error_handler_impl} and can be overridden to
      # change its behavior.
      #
      # @param error [SignalException]
      #
      def display_signal_notice(error)
        # Source available in the toys-core gem
      end

      ##
      # Displays a default output for an error. Displays the error, the
      # backtrace, and contextual information regarding what tool was run and
      # where in its code the error occurred.
      #
      # This method is used by {#error_handler_impl} and can be overridden to
      # change its behavior.
      #
      # @param error [Toys::ContextualError]
      #
      def display_error_notice(error)
        # Source available in the toys-core gem
      end

      ##
      # Implementation of the formatter used by loggers created by this UI's
      # logger factory. This interface is defined by the standard `Logger`
      # class.
      #
      # This method can be overridden to change the behavior of loggers created
      # by this UI.
      #
      # @param severity [String]
      # @param time [Time]
      # @param _progname [String]
      # @param msg [Object]
      # @return [String]
      #
      def logger_formatter_impl(severity, time, _progname, msg)
        # Source available in the toys-core gem
      end
    end
  end
end
