#!/bin/sh
set -e
# Repack legacy uqm-russian (0.6) addon to UQM 0.8 format.
# Usage: repack.sh input.zip output_dir
# orig = unpacked inner zip, new = plugin content. We mv files; at end, not-transferred.lst shows what stayed in orig.

REPACK_DIR="$(dirname "$0")"
CURDIR="$(pwd)"
ABS_REPACK="$(cd "$REPACK_DIR" && pwd)"

INPUT_ZIP="$1"
OUTPUT_DIR="$2"
test -r "$INPUT_ZIP"

rm -rf "$OUTPUT_DIR"
mkdir "$OUTPUT_DIR"
TMP_DIR="$OUTPUT_DIR/tmp-repack"
mkdir "$TMP_DIR"

# Unpack outer zip, extract russian.zip into orig
unzip -q "$INPUT_ZIP" -d "$TMP_DIR/outer"
[ -f "$TMP_DIR/outer/ChangeLog.txt" ] && cp "$TMP_DIR/outer/ChangeLog.txt" "$OUTPUT_DIR/ChangeLog.txt"
INNER_ZIP="$TMP_DIR/outer/russian.zip"
test -e "$INNER_ZIP"
unzip -q "$INNER_ZIP" -d "$TMP_DIR/orig"
rm -r "$TMP_DIR/outer"

ORIG="$TMP_DIR/orig"
NEW="$TMP_DIR/new"

# Normalize orig: put everything under base/
cd "$ORIG"
mkdir base
for d in *; do
  [ "$d" = base ] && continue
  [ -d "$d" ] && mv "$d" base/
done
cd "$CURDIR"

mkdir "$NEW"
mkdir "$NEW/base"

# 1) gamestrings
mv "$ORIG/base/lbm/starcon.txt" "$NEW/base/gamestrings.txt"

# 2) ipanims → official paths (lander/energy, lander/bio, lander, nav; and base/nav)
mkdir "$NEW/base/lander"
mkdir "$NEW/base/lander/energy"
mkdir "$NEW/base/lander/bio"
mkdir "$NEW/base/lander/nav"
mkdir "$NEW/base/nav"
mv "$ORIG/base/ipanims/moonbase.txt" "$NEW/base/lander/energy/moonbase.txt"
mv "$ORIG/base/ipanims/andruins.txt" "$NEW/base/lander/energy/androsynth_ruins.txt"
mv "$ORIG/base/ipanims/aquhelix.txt" "$NEW/base/lander/energy/aquahelix.txt"
mv "$ORIG/base/ipanims/burruins.txt" "$NEW/base/lander/energy/burvixeseruins.txt"
mv "$ORIG/base/ipanims/chmmrbas.txt" "$NEW/base/lander/energy/chmmrbase.txt"
mv "$ORIG/base/ipanims/drutrans.txt" "$NEW/base/lander/energy/fwiffo.txt"
mv "$ORIG/base/ipanims/eggcase.txt" "$NEW/base/lander/energy/eggcase.txt"
mv "$ORIG/base/ipanims/maidens.txt" "$NEW/base/lander/energy/maidens.txt"
mv "$ORIG/base/ipanims/ruins.txt" "$NEW/base/lander/energy/ruins.txt"
mv "$ORIG/base/ipanims/spapluto.txt" "$NEW/base/lander/energy/ultron.txt"
mv "$ORIG/base/ipanims/sphere.txt" "$NEW/base/lander/energy/sphere.txt"
mv "$ORIG/base/ipanims/spindle.txt" "$NEW/base/lander/energy/spindle.txt"
mv "$ORIG/base/ipanims/sundevic.txt" "$NEW/base/lander/energy/sundevice.txt"
mv "$ORIG/base/ipanims/syrvault.txt" "$NEW/base/lander/energy/syreenvault.txt"
mv "$ORIG/base/ipanims/taadevic.txt" "$NEW/base/lander/energy/taalodevice.txt"
mv "$ORIG/base/ipanims/umgtrans.txt" "$NEW/base/lander/energy/umgahcaster.txt"
mv "$ORIG/base/ipanims/urqwreck.txt" "$NEW/base/lander/energy/urquanwreck.txt"
mv "$ORIG/base/ipanims/utwbomb.txt" "$NEW/base/lander/energy/utwigbomb.txt"
mv "$ORIG/base/ipanims/vuxbeast.txt" "$NEW/base/lander/bio/vuxbeast.txt"
mv "$ORIG/base/ipanims/lander.32.png" "$NEW/base/lander/lander-032.png"
mv "$ORIG/base/ipanims/orbenter.0.png" "$NEW/base/nav/orbitenter-000.png"

# 3) lander font
mkdir "$NEW/base/fonts"
mv "$ORIG/base/ipanims/lander.fon" "$NEW/base/fonts/lander.fon"

# 4) comm: race dialogs. slyhome→slylandro, slyland→probe in 0.8.
# 5) Fonts to base/fonts/ (must run before move_comm). starbase, yehatrebels, safeones: no .fon in addon.
mkdir "$NEW/base/comm"
mv "$ORIG/base/comm/talkpet/talkpet.fon" "$NEW/base/fonts/talkingpet.fon"
mv "$ORIG/base/comm/slyhome/slyhome.fon" "$NEW/base/fonts/slylandro.fon"
mv "$ORIG/base/comm/slyland/slyland.fon" "$NEW/base/fonts/probe.fon"
mv "$ORIG/base/comm/melnorm/melnorm.fon" "$NEW/base/fonts/melnorme.fon"
mv "$ORIG/base/comm/pkunk/pkunk.fon" "$NEW/base/fonts/pkunk.fon"
mv "$ORIG/base/comm/zoqfot/zoqfot.fon" "$NEW/base/fonts/zoqfotpik.fon"
mv "$ORIG/base/comm/thradd/thradd.fon" "$NEW/base/fonts/thraddash.fon"
mv "$ORIG/base/comm/utwig/utwig.fon" "$NEW/base/fonts/utwig.fon"
mv "$ORIG/base/comm/syreen/syreen.fon" "$NEW/base/fonts/syreen.fon"
mv "$ORIG/base/comm/spathi/spathi.fon" "$NEW/base/fonts/spathi.fon"
mv "$ORIG/base/comm/ilwrath/ilwrath.fon" "$NEW/base/fonts/ilwrath.fon"
mv "$ORIG/base/comm/orz/orz.fon" "$NEW/base/fonts/orz.fon"
mv "$ORIG/base/comm/druuge/druuge.fon" "$NEW/base/fonts/druuge.fon"
mv "$ORIG/base/comm/vux/vux.fon" "$NEW/base/fonts/vux.fon"
mv "$ORIG/base/comm/comandr/comandr.fon" "$NEW/base/fonts/player.fon"
mv "$ORIG/base/comm/umgah/umgah.fon" "$NEW/base/fonts/umgah.fon"
mv "$ORIG/base/comm/arilou/arilou.fon" "$NEW/base/fonts/arilou.fon"
mv "$ORIG/base/comm/mycon/mycon.fon" "$NEW/base/fonts/mycon.fon"
mv "$ORIG/base/comm/supox/supox.fon" "$NEW/base/fonts/supox.fon"
mv "$ORIG/base/comm/urquan/urquan.fon" "$NEW/base/fonts/urquan.fon"
mv "$ORIG/base/comm/shofixt/shofixt.fon" "$NEW/base/fonts/shofixti.fon"
mv "$ORIG/base/comm/blackur/blackur.fon" "$NEW/base/fonts/kohrah.fon"
mv "$ORIG/base/comm/yehat/yehat.fon" "$NEW/base/fonts/yehat.fon"
mv "$ORIG/base/comm/chmmr/chmmr.fon" "$NEW/base/fonts/chmmr.fon"

move_comm() {
  old="$1"; new="$2"
  mv "$ORIG/base/comm/$old" "$NEW/base/comm/$new"
  if [ "$old" != "$new" ]; then
    mv "$NEW/base/comm/$new/$old.txt" "$NEW/base/comm/$new/$new.txt"
  fi
}
move_comm starbas starbase
move_comm rebel yehatrebels
move_comm talkpet talkingpet
move_comm spahome safeones
move_comm slyhome slylandro
move_comm slyland probe
move_comm melnorm melnorme
move_comm pkunk pkunk
move_comm zoqfot zoqfotpik
move_comm thradd thraddash
move_comm utwig utwig
move_comm syreen syreen
move_comm spathi spathi
move_comm ilwrath ilwrath
move_comm orz orz
move_comm druuge druuge
move_comm vux vux
move_comm comandr commander
move_comm umgah umgah
move_comm arilou arilou
move_comm mycon mycon
move_comm supox supox
move_comm urquan urquan
move_comm shofixt shofixti
move_comm blackur kohrah
move_comm yehat yehat
move_comm chmmr chmmr

# 6) Fonts from lbm and slides
mv "$ORIG/base/lbm/micro.fon" "$NEW/base/fonts/micro.fon"
mv "$ORIG/base/lbm/starcon.fon" "$NEW/base/fonts/starcon.fon"
mv "$ORIG/base/lbm/tiny.fon" "$NEW/base/fonts/tiny.fon"
mv "$ORIG/base/slides/slides.fon" "$NEW/base/fonts/slides.fon"

# 6a) slides → cutscene
mkdir "$NEW/base/cutscene"
mkdir "$NEW/base/cutscene/ending"
mkdir "$NEW/base/cutscene/intro"
mv "$ORIG/base/slides/ending/final.txt" "$NEW/base/cutscene/ending/final.txt"
mv "$ORIG/base/slides/intro/intro.txt" "$NEW/base/cutscene/intro/intro.txt"

# 7) Rename numeric glyph filenames in fonts: decimal → hex (e.g. 10.png → a.png)
for fon_dir in "$NEW/base/fonts"/*.fon; do
  [ -d "$fon_dir" ] || continue
  cd "$fon_dir"
  for f in *.png; do
    [ -f "$f" ] || continue
    b="${f%.*}"
    case "$b" in
      [0-9]*) new=$(printf "%x.png" "$b"); [ "$f" != "$new" ] && mv "$f" "$new" ;;
    esac
  done
  for f in *.lbm; do
    [ -f "$f" ] || continue
    b="${f%.*}"
    case "$b" in
      [0-9]*) new=$(printf "%x.lbm" "$b"); [ "$f" != "$new" ] && mv "$f" "$new" ;;
    esac
  done
  cd "$CURDIR"
done

# 8) Ship description texts: orig/base/<race>/*text.txt → base/ships/<race>/<ship>.txt
mkdir -p "$NEW/base/ships/shofixti" "$NEW/base/ships/spathi" "$NEW/base/ships/arilou" "$NEW/base/ships/zoqfotpik" "$NEW/base/ships/yehat" "$NEW/base/ships/vux" "$NEW/base/ships/utwig" "$NEW/base/ships/urquan" "$NEW/base/ships/umgah" \
  "$NEW/base/ships/thraddash" "$NEW/base/ships/syreen" "$NEW/base/ships/supox" "$NEW/base/ships/pkunk" "$NEW/base/ships/orz" "$NEW/base/ships/mycon" "$NEW/base/ships/mmrnmhrm" "$NEW/base/ships/melnorme" "$NEW/base/ships/ilwrath" "$NEW/base/ships/human" "$NEW/base/ships/druuge" "$NEW/base/ships/chmmr" "$NEW/base/ships/chenjesu" "$NEW/base/ships/kohrah" "$NEW/base/ships/androsynth" "$NEW/base/ships/slylandro"
mv "$ORIG/base/shofixti/shotext.txt" "$NEW/base/ships/shofixti/scout.txt"
mv "$ORIG/base/spathi/spatext.txt" "$NEW/base/ships/spathi/eluder.txt"
mv "$ORIG/base/arilou/aritext.txt" "$NEW/base/ships/arilou/skiff.txt"
mv "$ORIG/base/zoqfot/zoqtext.txt" "$NEW/base/ships/zoqfotpik/stinger.txt"
mv "$ORIG/base/yehat/yehtext.txt" "$NEW/base/ships/yehat/terminator.txt"
mv "$ORIG/base/vux/vuxtext.txt" "$NEW/base/ships/vux/intruder.txt"
mv "$ORIG/base/utwig/utwtext.txt" "$NEW/base/ships/utwig/jugger.txt"
mv "$ORIG/base/urquan/urqtext.txt" "$NEW/base/ships/urquan/dreadnought.txt"
mv "$ORIG/base/umgah/umgtext.txt" "$NEW/base/ships/umgah/drone.txt"
mv "$ORIG/base/thradd/thrtext.txt" "$NEW/base/ships/thraddash/torch.txt"
mv "$ORIG/base/syreen/syrtext.txt" "$NEW/base/ships/syreen/penetrator.txt"
mv "$ORIG/base/supox/suptext.txt" "$NEW/base/ships/supox/blade.txt"
mv "$ORIG/base/pkunk/pkutext.txt" "$NEW/base/ships/pkunk/fury.txt"
mv "$ORIG/base/orz/orztext.txt" "$NEW/base/ships/orz/nemesis.txt"
mv "$ORIG/base/mycon/myctext.txt" "$NEW/base/ships/mycon/podship.txt"
mv "$ORIG/base/mmrnmhrm/mmrtext.txt" "$NEW/base/ships/mmrnmhrm/xform.txt"
mv "$ORIG/base/melnorme/meltext.txt" "$NEW/base/ships/melnorme/trader.txt"
mv "$ORIG/base/ilwrath/ilwtext.txt" "$NEW/base/ships/ilwrath/avenger.txt"
mv "$ORIG/base/human/humtext.txt" "$NEW/base/ships/human/cruiser.txt"
mv "$ORIG/base/druuge/drutext.txt" "$NEW/base/ships/druuge/mauler.txt"
mv "$ORIG/base/chmmr/chmtext.txt" "$NEW/base/ships/chmmr/avatar.txt"
mv "$ORIG/base/chenjesu/chetext.txt" "$NEW/base/ships/chenjesu/broodhome.txt"
mv "$ORIG/base/blackurq/blatext.txt" "$NEW/base/ships/kohrah/marauder.txt"
mv "$ORIG/base/androsyn/andtext.txt" "$NEW/base/ships/androsynth/guardian.txt"
mv "$ORIG/base/slylandr/slytext.txt" "$NEW/base/ships/slylandro/probe.txt"

# 8a) melee → base/ui (melebkgd.N → meleemenu-NNN, netplay.N → netplay-NNN)
mkdir "$NEW/base/ui"
for n in 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30; do
  nn=$(printf "%03d" "$n")
  mv "$ORIG/base/melee/melebkgd.$n.png" "$NEW/base/ui/meleemenu-$nn.png"
done
for n in 0 1 2 3 4 5; do
  nn=$(printf "%03d" "$n")
  mv "$ORIG/base/melee/netplay.$n.png" "$NEW/base/ui/netplay-$nn.png"
done

# 8a2) base/lbm/ → base/ui/ and base/cutscene/credits/: title, credits, activity, playmenu, shipyard, modules, outfitmodules
mkdir "$NEW/base/cutscene/credits"
mv "$ORIG/base/lbm/title.png" "$NEW/base/ui/title.png"
# credits: credit01.0.png..credit19.0.png → credback-001.png..credback-019.png
mv "$ORIG/base/lbm/credit01.0.png" "$NEW/base/cutscene/credits/credback-001.png"
mv "$ORIG/base/lbm/credit02.0.png" "$NEW/base/cutscene/credits/credback-002.png"
mv "$ORIG/base/lbm/credit03.0.png" "$NEW/base/cutscene/credits/credback-003.png"
mv "$ORIG/base/lbm/credit04.0.png" "$NEW/base/cutscene/credits/credback-004.png"
mv "$ORIG/base/lbm/credit05.0.png" "$NEW/base/cutscene/credits/credback-005.png"
mv "$ORIG/base/lbm/credit06.0.png" "$NEW/base/cutscene/credits/credback-006.png"
mv "$ORIG/base/lbm/credit07.0.png" "$NEW/base/cutscene/credits/credback-007.png"
mv "$ORIG/base/lbm/credit08.0.png" "$NEW/base/cutscene/credits/credback-008.png"
mv "$ORIG/base/lbm/credit09.0.png" "$NEW/base/cutscene/credits/credback-009.png"
mv "$ORIG/base/lbm/credit10.0.png" "$NEW/base/cutscene/credits/credback-010.png"
mv "$ORIG/base/lbm/credit11.0.png" "$NEW/base/cutscene/credits/credback-011.png"
mv "$ORIG/base/lbm/credit12.0.png" "$NEW/base/cutscene/credits/credback-012.png"
mv "$ORIG/base/lbm/credit13.0.png" "$NEW/base/cutscene/credits/credback-013.png"
mv "$ORIG/base/lbm/credit14.0.png" "$NEW/base/cutscene/credits/credback-014.png"
mv "$ORIG/base/lbm/credit15.0.png" "$NEW/base/cutscene/credits/credback-015.png"
mv "$ORIG/base/lbm/credit16.0.png" "$NEW/base/cutscene/credits/credback-016.png"
mv "$ORIG/base/lbm/credit17.0.png" "$NEW/base/cutscene/credits/credback-017.png"
mv "$ORIG/base/lbm/credit18.0.png" "$NEW/base/cutscene/credits/credback-018.png"
mv "$ORIG/base/lbm/credit19.0.png" "$NEW/base/cutscene/credits/credback-019.png"
{
  echo "credback-001.png -1 -1 0 0"
  echo "credback-002.png -1 -1 0 0"
  echo "credback-003.png -1 -1 0 0"
  echo "credback-004.png -1 -1 0 0"
  echo "credback-005.png -1 -1 0 0"
  echo "credback-006.png -1 -1 0 0"
  echo "credback-007.png -1 -1 0 0"
  echo "credback-008.png -1 -1 0 0"
  echo "credback-009.png -1 -1 0 0"
  echo "credback-010.png -1 -1 0 0"
  echo "credback-011.png -1 -1 0 0"
  echo "credback-012.png -1 -1 0 0"
  echo "credback-013.png -1 -1 0 0"
  echo "credback-014.png -1 -1 0 0"
  echo "credback-015.png -1 -1 0 0"
  echo "credback-016.png -1 -1 0 0"
  echo "credback-017.png -1 -1 0 0"
  echo "credback-018.png -1 -1 0 0"
  echo "credback-019.png -1 -1 0 0"
} > "$NEW/base/cutscene/credits/credback.ani"
# activity (2 frames)
mv "$ORIG/base/lbm/activity.0.png" "$NEW/base/ui/activity-000.png"
mv "$ORIG/base/lbm/activity.1.png" "$NEW/base/ui/activity-001.png"
# playmenu (67 frames): .N → -NNN
mv "$ORIG/base/lbm/playmenu.0.png" "$NEW/base/ui/playmenu-000.png"
mv "$ORIG/base/lbm/playmenu.1.png" "$NEW/base/ui/playmenu-001.png"
mv "$ORIG/base/lbm/playmenu.2.png" "$NEW/base/ui/playmenu-002.png"
mv "$ORIG/base/lbm/playmenu.3.png" "$NEW/base/ui/playmenu-003.png"
mv "$ORIG/base/lbm/playmenu.4.png" "$NEW/base/ui/playmenu-004.png"
mv "$ORIG/base/lbm/playmenu.5.png" "$NEW/base/ui/playmenu-005.png"
mv "$ORIG/base/lbm/playmenu.6.png" "$NEW/base/ui/playmenu-006.png"
mv "$ORIG/base/lbm/playmenu.7.png" "$NEW/base/ui/playmenu-007.png"
mv "$ORIG/base/lbm/playmenu.8.png" "$NEW/base/ui/playmenu-008.png"
mv "$ORIG/base/lbm/playmenu.9.png" "$NEW/base/ui/playmenu-009.png"
mv "$ORIG/base/lbm/playmenu.10.png" "$NEW/base/ui/playmenu-010.png"
mv "$ORIG/base/lbm/playmenu.11.png" "$NEW/base/ui/playmenu-011.png"
mv "$ORIG/base/lbm/playmenu.12.png" "$NEW/base/ui/playmenu-012.png"
mv "$ORIG/base/lbm/playmenu.13.png" "$NEW/base/ui/playmenu-013.png"
mv "$ORIG/base/lbm/playmenu.14.png" "$NEW/base/ui/playmenu-014.png"
mv "$ORIG/base/lbm/playmenu.15.png" "$NEW/base/ui/playmenu-015.png"
mv "$ORIG/base/lbm/playmenu.16.png" "$NEW/base/ui/playmenu-016.png"
mv "$ORIG/base/lbm/playmenu.17.png" "$NEW/base/ui/playmenu-017.png"
mv "$ORIG/base/lbm/playmenu.18.png" "$NEW/base/ui/playmenu-018.png"
mv "$ORIG/base/lbm/playmenu.19.png" "$NEW/base/ui/playmenu-019.png"
mv "$ORIG/base/lbm/playmenu.20.png" "$NEW/base/ui/playmenu-020.png"
mv "$ORIG/base/lbm/playmenu.21.png" "$NEW/base/ui/playmenu-021.png"
mv "$ORIG/base/lbm/playmenu.22.png" "$NEW/base/ui/playmenu-022.png"
mv "$ORIG/base/lbm/playmenu.23.png" "$NEW/base/ui/playmenu-023.png"
mv "$ORIG/base/lbm/playmenu.24.png" "$NEW/base/ui/playmenu-024.png"
mv "$ORIG/base/lbm/playmenu.25.png" "$NEW/base/ui/playmenu-025.png"
mv "$ORIG/base/lbm/playmenu.26.png" "$NEW/base/ui/playmenu-026.png"
mv "$ORIG/base/lbm/playmenu.27.png" "$NEW/base/ui/playmenu-027.png"
mv "$ORIG/base/lbm/playmenu.28.png" "$NEW/base/ui/playmenu-028.png"
mv "$ORIG/base/lbm/playmenu.29.png" "$NEW/base/ui/playmenu-029.png"
mv "$ORIG/base/lbm/playmenu.30.png" "$NEW/base/ui/playmenu-030.png"
mv "$ORIG/base/lbm/playmenu.31.png" "$NEW/base/ui/playmenu-031.png"
mv "$ORIG/base/lbm/playmenu.32.png" "$NEW/base/ui/playmenu-032.png"
mv "$ORIG/base/lbm/playmenu.33.png" "$NEW/base/ui/playmenu-033.png"
mv "$ORIG/base/lbm/playmenu.34.png" "$NEW/base/ui/playmenu-034.png"
mv "$ORIG/base/lbm/playmenu.35.png" "$NEW/base/ui/playmenu-035.png"
mv "$ORIG/base/lbm/playmenu.36.png" "$NEW/base/ui/playmenu-036.png"
mv "$ORIG/base/lbm/playmenu.37.png" "$NEW/base/ui/playmenu-037.png"
# playmenu.38–57 not in addon
mv "$ORIG/base/lbm/playmenu.58.png" "$NEW/base/ui/playmenu-058.png"
mv "$ORIG/base/lbm/playmenu.59.png" "$NEW/base/ui/playmenu-059.png"
mv "$ORIG/base/lbm/playmenu.60.png" "$NEW/base/ui/playmenu-060.png"
mv "$ORIG/base/lbm/playmenu.61.png" "$NEW/base/ui/playmenu-061.png"
mv "$ORIG/base/lbm/playmenu.62.png" "$NEW/base/ui/playmenu-062.png"
mv "$ORIG/base/lbm/playmenu.63.png" "$NEW/base/ui/playmenu-063.png"
mv "$ORIG/base/lbm/playmenu.64.png" "$NEW/base/ui/playmenu-064.png"
mv "$ORIG/base/lbm/playmenu.65.png" "$NEW/base/ui/playmenu-065.png"
mv "$ORIG/base/lbm/playmenu.66.png" "$NEW/base/ui/playmenu-066.png"
# Do not ship playmenu.ani from addon: it references all 0–66 but we only have 0–37 and 58–66 (38–57 missing in addon). Game will use base content .ani; our PNGs override frames we provide.
# shipyard: only frames we have
mv "$ORIG/base/lbm/shipyard.3.png" "$NEW/base/ui/shipyard-003.png"
mv "$ORIG/base/lbm/shipyard.4.png" "$NEW/base/ui/shipyard-004.png"
mv "$ORIG/base/lbm/shipyard.5.png" "$NEW/base/ui/shipyard-005.png"
mv "$ORIG/base/lbm/shipyard.6.png" "$NEW/base/ui/shipyard-006.png"
mv "$ORIG/base/lbm/shipyard.7.png" "$NEW/base/ui/shipyard-007.png"
mv "$ORIG/base/lbm/shipyard.8.png" "$NEW/base/ui/shipyard-008.png"
mv "$ORIG/base/lbm/shipyard.9.png" "$NEW/base/ui/shipyard-009.png"
mv "$ORIG/base/lbm/shipyard.10.png" "$NEW/base/ui/shipyard-010.png"
mv "$ORIG/base/lbm/shipyard.11.png" "$NEW/base/ui/shipyard-011.png"
mv "$ORIG/base/lbm/shipyard.12.png" "$NEW/base/ui/shipyard-012.png"
mv "$ORIG/base/lbm/shipyard.14.png" "$NEW/base/ui/shipyard-014.png"
mv "$ORIG/base/lbm/shipyard.15.png" "$NEW/base/ui/shipyard-015.png"
mv "$ORIG/base/lbm/shipyard.16.png" "$NEW/base/ui/shipyard-016.png"
mv "$ORIG/base/lbm/shipyard.20.png" "$NEW/base/ui/shipyard-020.png"
mv "$ORIG/base/lbm/shipyard.22.png" "$NEW/base/ui/shipyard-022.png"
mv "$ORIG/base/lbm/shipyard.23.png" "$NEW/base/ui/shipyard-023.png"
# shipyard.ani (if addon has it)
awk '{ for(i=0;i<=99;i++) gsub("shipyard." i "\\.png", "shipyard-" sprintf("%03d",i) ".png"); print }' "$ORIG/base/lbm/shipyard.ani" > "$NEW/base/ui/shipyard.ani" 2>/dev/null || true
# modules (ship modules menu): modules.0..15 → modulesmenu-000..015
mv "$ORIG/base/lbm/modules.0.png" "$NEW/base/ui/modulesmenu-000.png"
mv "$ORIG/base/lbm/modules.1.png" "$NEW/base/ui/modulesmenu-001.png"
mv "$ORIG/base/lbm/modules.2.png" "$NEW/base/ui/modulesmenu-002.png"
mv "$ORIG/base/lbm/modules.3.png" "$NEW/base/ui/modulesmenu-003.png"
mv "$ORIG/base/lbm/modules.4.png" "$NEW/base/ui/modulesmenu-004.png"
mv "$ORIG/base/lbm/modules.5.png" "$NEW/base/ui/modulesmenu-005.png"
mv "$ORIG/base/lbm/modules.6.png" "$NEW/base/ui/modulesmenu-006.png"
mv "$ORIG/base/lbm/modules.7.png" "$NEW/base/ui/modulesmenu-007.png"
mv "$ORIG/base/lbm/modules.8.png" "$NEW/base/ui/modulesmenu-008.png"
mv "$ORIG/base/lbm/modules.9.png" "$NEW/base/ui/modulesmenu-009.png"
mv "$ORIG/base/lbm/modules.10.png" "$NEW/base/ui/modulesmenu-010.png"
mv "$ORIG/base/lbm/modules.11.png" "$NEW/base/ui/modulesmenu-011.png"
mv "$ORIG/base/lbm/modules.12.png" "$NEW/base/ui/modulesmenu-012.png"
mv "$ORIG/base/lbm/modules.13.png" "$NEW/base/ui/modulesmenu-013.png"
mv "$ORIG/base/lbm/modules.14.png" "$NEW/base/ui/modulesmenu-014.png"
mv "$ORIG/base/lbm/modules.15.png" "$NEW/base/ui/modulesmenu-015.png"
# sismods.55.png → outfitmodules-055.png (one module icon)
mv "$ORIG/base/lbm/sismods.55.png" "$NEW/base/ui/outfitmodules-055.png"

# 8b) lastbat, sis_ship: not transferred — in 0.8 these are animations; addon had single/wrong-size frames and would break the ship panel.

# 9) setupmenu and main menu. setupmenu from repack-0.8; newgame from orig/base/lbm/ (rename .N.png → -NNN.png, fix frame names in .ani)
rm "$ORIG/base/lbm/setupmenu.txt"
cp "$ABS_REPACK/setupmenu-ru.txt" "$NEW/base/ui/setupmenu.txt"
for n in 0 1 2 3 4 5; do
  nn=$(printf "%03d" "$n")
  mv "$ORIG/base/lbm/newgame.$n.png" "$NEW/base/ui/newgame-$nn.png"
done
sed 's/newgame\.\([0-5]\)\.png/newgame-00\1.png/g' "$ORIG/base/lbm/newgame.ani" > "$NEW/base/ui/newgame.ani"

# Apply patches from patches/ (gamestrings, safeones, etc.)
PATCHES_DIR="$ABS_REPACK/patches"
for p in "$PATCHES_DIR"/*.patch; do
  [ -f "$p" ] || continue
  (cd "$NEW" && patch -p1 < "$p") || { echo "repack.sh: patch failed: $p" >&2; exit 1; }
done

# Build plugin from new
ZIP_ABS="$CURDIR/$OUTPUT_DIR/tmp-repack/russian/shadow-content/russian-content.zip"
mkdir "$TMP_DIR/russian"
mkdir "$TMP_DIR/russian/shadow-content"
(cd "$NEW" && zip -r -q -0 "$ZIP_ABS" base/)
printf '# Russian translation addon - overrides base via shadow-content\n' > "$TMP_DIR/russian/russian.rmp"
(cd "$CURDIR" && cd "$TMP_DIR" && zip -r -q -0 "$CURDIR/$OUTPUT_DIR/uqm-0.8.0-russian.uqm" russian/)

# ChangeLog from orig if present
for cl in "$ORIG/ChangeLog.txt" "$ORIG/base/ChangeLog.txt"; do
  [ -f "$cl" ] && cp "$cl" "$OUTPUT_DIR/ChangeLog.txt" && break
done

# Plugin contents and list of files not transferred (left in orig)
(cd "$NEW" && find base -type f | sort) > "$OUTPUT_DIR/contents-base.lst"
(find "$ORIG" -type f | sort) > "$OUTPUT_DIR/not-transferred.lst"

echo "repack.sh: $OUTPUT_DIR/uqm-0.8.0-russian.uqm created."
echo "  base contents:   $OUTPUT_DIR/contents-base.lst"
echo "  not transferred: $OUTPUT_DIR/not-transferred.lst"
